package com.joyy.hagorpc.internal

import com.joyy.hagorpc.IRPCDataProvider
import com.joyy.hagorpc.RPCCallException
import com.joyy.hagorpc.RPCCallRequest
import com.joyy.hagorpc.RPCCallResponse
import com.joyy.hagorpc.RPCPacketV2
import com.joyy.hagorpc.ResponseBundle
import com.joyy.hagorpc.birdge.Base64Bridge
import com.joyy.hagorpc.birdge.ExecutorBridge
import com.joyy.hagorpc.birdge.HttpClientBridge
import com.joyy.hagorpc.birdge.LoggerBridge
import com.joyy.hagorpc.birdge.RPCPacketBridge
import com.joyy.hagorpc.birdge.SystemBridge
import com.joyy.hagorpc.birdge.URLEncoderBridge


internal data class RPCRequest(
    val httpUrl: String,
    var wsUrl: String,
    val packet: RPCPacketV2,
    val routingKey: String?,
    private val callback: RPCCore.RPCCallback,
    val checkByHttp: Boolean = false
) {

    companion object {
        private const val TAG = "RPCCore.RPCRequest"
    }

    var length: Long = 0 // 请求body的长度
    var retryCount: Int = 0
    var oldSeqId: Long = 0 // 上次的请求序列号
    var seqId: Long = 0 // 最新的请求序列号
    var queueSize: Long = -1L //当前所述队列数（分长连接和短连接）
    var isHttpRequesting: Boolean = false // 当前是否用http短连接请求中（特定情况可以使用短连接发送）

    // 发送是否成功
    var sendSuccess: Boolean = false

    // 用于统计的跟踪参数
    var trace: String = ""

    // 发送时间戳 ms
    var sendTime: Long = SystemBridge.currentTimeMillis()
    var call: RPCCallRequest? = null // 记录的请求数据

    /**
     * 重新发送请求
     */
    fun resend() {
        retryCount++
        LoggerBridge.logI(TAG, "resend increment retryCount：$retryCount, seqId: $seqId")
        send()
    }

    fun send(): Boolean {
        val queueSizePair = RPCCore.getRequestQueueSize(this)
        var dispatchResult: IRPCDataProvider.DispatchResult? = null
        if (checkByHttp) {
            val condition = IRPCDataProvider.DispatchCondition(
                packet.header.sname, packet.header.method,
                callback.expectHttpRequest(),
                callback.needToken(),
                queueSizePair.first
            )
            dispatchResult = RPCCore.mDispatchRequestToHttp(condition)
        }

        val useHttp = dispatchResult?.useHttp == true
        val waitTimeout =
            if (dispatchResult?.useWaitTimeout == true) callback.getTimeout() else 0

        val msg = if (retryCount <= 0) {
            "send request: $this"
        } else {
            "resend request: $this"
        }
        LoggerBridge.logD(
            TAG,
            "$msg, current queue: ${queueSizePair.first + queueSizePair.second + 1}"
        )
        sendTime = SystemBridge.currentTimeMillis()
        sendSuccess = if (useHttp) {
            isHttpRequesting = true
            queueSize = queueSizePair.second.toLong()
            // 重试场景也要回调
            callbackBeforeRequest()
            httpSend(waitTimeout)
        } else {
            isHttpRequesting = false
            queueSize = RPCCore.mWebSocket?.queueSize() ?: 0
            // 重试场景也要回调
            callbackBeforeRequest()
            wsSend()
        }
        return sendSuccess
    }

    internal fun onSuccess(success: ResponseBundle) {
        this.trace += "_response"
        recycle()
        val result = RPCCallResponse(
            RPCCallRequest(this.call, this.trace),
            success.responseBytes, success.payload,
            success.responseTimestamp, success.dispatchTimestamp,
            success.responseHeader
        )
        callback.onSuccess(result)
    }

    internal fun onError(
        canRetry: Boolean = false,
        fromTimeout: Boolean = false,
        code: Int = -1,
        reason: String = "",
        responseHeader: Map<String, List<String>> = emptyMap()
    ): Boolean {
        trace += if (fromTimeout) {
            "_onTimeout"
        } else {
            "_onError"
        }
        val error = RPCCallException(
            RPCCallRequest(this.call, this.trace),
            canRetry, fromTimeout, reason, code, responseHeader
        )
        val wantRetry = callback.onError(error)
        if (!wantRetry || !canRetry) {
            // 如果上层不希望重试或当前不允许重试，则此次请求不再有必要，回收
            recycle()
        }
        return wantRetry
    }

    internal fun getTimeout(): Long {
        return callback.getTimeout()
    }

    /**
     * 更新请求序列号(同时通知调用方)
     */
    internal fun updateSequenceId(newId: Long) {
        this.oldSeqId = this.seqId
        this.seqId = newId
        LoggerBridge.logD(TAG, "updateRequest oldSeqId: $oldSeqId, newSeqId: $seqId")
        // 添加或更新到map中
        val requestMap = RPCCore.mRequestMap.value
        requestMap.remove(oldSeqId)
        requestMap[seqId] = this
    }

    /**
     * 发送请求前回调通知
     */
    private fun callbackBeforeRequest() {
        if (retryCount <= 0) {
            this.trace += "_add"
        } else {
            this.trace += "_resend"
        }
        val call = RPCCallRequest(
            if (isHttpRequesting) {
                this.httpUrl
            } else {
                this.wsUrl
            },
            this.oldSeqId,
            this.seqId,
            this.packet.header.sname,
            this.packet.header.method,
            this.packet.uri,
            this.length,
            this.sendTime,
            this.retryCount,
            this.queueSize,
            this.isHttpRequesting,
            this.trace
        )
        this.call = call
        LoggerBridge.logD(TAG, "callback before request: $call")
        callback.onRequest(call)
        if (this.retryCount <= 0) {
            ExecutorBridge.getMainExecutor().execute {
                RPCCore.mEventListener.onStartRequest(call)
            }
        }
    }

    private fun wsSend(): Boolean {
        val bs: ByteArray = packet.toByteArray.invoke()
        if (bs.isEmpty()) {
            return false
        }
        return if (!RPCCore.checkConnected()) {
            false
        } else RPCCore.mWebSocket?.send(bs) ?: false
    }

    private fun httpSend(waitTimeout: Long): Boolean {
        val sName = packet.header.sname
        val method = packet.header.method
        val requestHeader = mutableMapOf<String, String>()
        requestHeader.putAll(RPCCore.mCommonHeader.invoke(wsUrl, false))
        requestHeader[RPCConst.X_AUTH_TOKEN] = RPCCore.mToken.invoke()
        requestHeader[RPCConst.X_ROOM_ID] = packet.header.roomid
        requestHeader[RPCConst.X_YMICRO_API_SERVICE_NAME] = URLEncoderBridge.encode(sName)
        requestHeader[RPCConst.X_YMICRO_API_METHOD_NAME] = URLEncoderBridge.encode(method)
        // 标记路由规则header
        var finalRoutingKey = routingKey
        if (finalRoutingKey.isNullOrEmpty()) {
            finalRoutingKey = RPCLocalPersistence.getCProxyRouting(sName)
            if (finalRoutingKey.isNotEmpty()) {
                finalRoutingKey = Base64Bridge.encode(finalRoutingKey)
            }
        }
        if (finalRoutingKey.isNotEmpty()) {
            requestHeader[RPCConst.X_ROUTE_KEY_NAME] = finalRoutingKey
        }
        requestHeader["Content-Type"] = "application/protobuf"
        return HttpClientBridge.send(
            httpUrl = httpUrl,
            method = HttpClientBridge.HttpMethod.POST,
            requestHeader = requestHeader,
            payload = packet.payload,
            waitTimeout = waitTimeout,
            onResponse = { responseBytes: ByteArray, responseLength: Long, responseHeader: Map<String, List<String>> ->
                // 从头部获取route key更新
                updateRoutingKeyFromHeader(sName, responseHeader)
                onSuccess(
                    ResponseBundle(
                        protocol = packet,
                        responseBytes = responseBytes,
                        payload = responseLength,
                        responseHeader = responseHeader
                    )
                )
            },
            onError = { canRetry: Boolean, fromTimeout: Boolean, code: Int, reason: String, responseHeader: Map<String, List<String>> ->
                var needRetry = canRetry
                if (canRetry && retryCount >= RPCCore.mRequestMaxRetry) {
                    // 重试次数达到上限
                    needRetry = false
                }
                val toRetry = onError(
                    canRetry,
                    fromTimeout,
                    code,
                    reason,
                    responseHeader
                )
                if (needRetry && toRetry) {
                    resendRequest(this)
                }
            }
        )
    }

    private fun resendRequest(request: RPCRequest) {
        ExecutorBridge.getAsyncExecutor().execute {
            request.resend()
        }
    }

    private fun updateRoutingKeyFromHeader(service: String?, header: Map<String, List<String>>?) {
        if (service.isNullOrEmpty() || header.isNullOrEmpty()) {
            return
        }
        val list = header[RPCConst.X_ROUTE_KEY_NAME]
        if (list != null && list.isNotEmpty()) {
            var key = list[0]
            if (key.isNotEmpty()) {
                key = Base64Bridge.decode(key)
            }

            RPCLocalPersistence.updateCProxyRouting(service, key)
        }
    }

    private fun recycle() {
        LoggerBridge.logD(TAG, "recycle reqId: $seqId, trace: $trace")
        // request被回收，抛出事件，并从map移除
        call?.let {
            ExecutorBridge.getMainExecutor().execute {
                RPCCore.mEventListener.onFinishRequest(RPCCallRequest(it, trace))
            }
        }
        val requestMap = RPCCore.mRequestMap.value
        requestMap.remove(seqId)
    }
}
