package com.joyy.hagorpc.internal

import com.google.protobuf.GeneratedMessageLite
import com.joyy.hagorpc.ILogger
import com.joyy.hagorpc.impl.DefaultLogger
import com.squareup.wire.AndroidMessage
import com.squareup.wire.ProtoAdapter
import common.Header
import common.Response
import common.Result
import ikxd.cproxy.InnerV2
import ikxd.cproxy.InnerV3
import okio.ByteString.Companion.toByteString
import java.lang.reflect.ParameterizedType
import java.lang.reflect.Type
import java.net.*
import javax.net.ssl.SSLHandshakeException

/**
 * Created by chenrenzhan on 2018/9/20.
 */
object RPCProtoHelper {
    private const val TAG = "ProtoUtils"

    @Volatile
    private var sLogger: ILogger = DefaultLogger()
    fun setLogger(logger: ILogger) {
        sLogger = logger
    }

    fun isNetworkException(throwable: Throwable?): Boolean {
        return throwable is UnknownHostException ||
                throwable is NoRouteToHostException ||
                throwable is ConnectException ||
                throwable is SocketTimeoutException ||
                throwable is SSLHandshakeException
    }

    fun urlEncoderEncode(url: String): String {
        try {
            return URLEncoder.encode(url, "UTF-8")
        } catch (e: Exception) {
            sLogger.logE(TAG, "urlEncoderEncode", e)
        }
        return url
    }

    fun parseInner(bs: ByteArray): InnerV2? {
        if (bs.isEmpty()) {
            return null
        }
        var inner: InnerV2? = null
        try {
            val copyBs = bs.copyOf(bs.size)
            inner = InnerV2.ADAPTER.decode(copyBs)
        } catch (e: Exception) {
            sLogger.logE(TAG, "", e)
        }
        return inner
    }

    fun parseInnerV3(bs: ByteArray): InnerV3? {
        if (bs.isEmpty()) {
            return null
        }
        var inner: InnerV3? = null
        try {
            val copyBs = bs.copyOf(bs.size)
            inner = InnerV3.ADAPTER.decode(copyBs)
        } catch (e: Exception) {
            sLogger.logE(TAG, "", e)
        }
        return inner
    }

    fun headerString(header: Header?): String {
        return if (header == null) {
            "header is null"
        } else ToolUtil
            .format(
                "name: %s, method: %s, seq: %d, lang: %s, code: %d, version: %s, msgtype: %s, " +
                        "extend: %d",
                header.sname, header.method, header.seqid, header.lang,
                header.code, header.version, header.msgtype, header.extend.size
            )
    }

    fun innerString(inner: InnerV2?): String? {
        if (inner == null) {
            return "inner: null"
        }
        val header = inner.header
            ?: return ToolUtil.formatWithUSLocal("header is null, uri: %d", inner.uri)
        return ToolUtil.format(
            "name: %s, method: %s, uri: %d, seq: %d, lang: %s, code: %d, version: %s, msgtype: %s, roomId: %s, " +
                    "extend: %d",
            header.sname, header.method, inner.uri, header.seqid, header.lang,
            header.code, header.version, header.msgtype, header.roomid,
            header.extend.size
        )
    }

    @JvmStatic
    fun getHeaderBuilder(sName: String, v2: Boolean): Header.Builder {
        val result = RPCCore.getHeaderBuilder(sName, v2)
        return Header.Builder()
            .sname(result.sname)
            .seqid(result.seqid)
            .code(result.code)
            .lang(result.lang)
            .tid(result.tid)
            .roomid(result.roomid)
            .method(result.method)
            .version(result.version)
            .msgtype(Header.MSGTYPE.fromValue(result.msgtype.value))
            .back_ground(result.backGround)
            .biz(result.biz?.toByteString())
            .extend(result.extend.mapValues { it.value.toByteString() })
            .routing_key(result.routingKey?.toByteString())
            .gameid(result.gameid)
    }

    /**
     * 这里拿到的type 可能会丢失 RES 的类型信息, 例如 generic method
     *
     *
     * public static <T extends GeneratedMessageLite></T><T></T>, ?>> T of() {
     * MyProtoCallback<T> callback = new MyProtoCallback<>();
     * }
    </T> */
    fun <T : Any> getInterfaceType(inter: T, cls: Class<T>): Type? {
        for (type in inter::class.java.genericInterfaces) {
            if (type is ParameterizedType &&
                cls.isAssignableFrom(type.rawType as Class<*>)
            ) {
                return type
            }
        }
        val type = inter.javaClass.genericSuperclass
        return if (type is ParameterizedType &&
            cls.isAssignableFrom(type.rawType as Class<*>)
        ) {
            type
        } else null
    }

    fun <T : GeneratedMessageLite<T, *>> parseBytes(
        defaultInstance: T,
        data: ByteArray
    ): T? {
        try {
            return _parseBytes(defaultInstance, data)
        } catch (e: Exception) {
            sLogger.logE(TAG, "", e)
        }
        return null
    }

    fun resultString(resultRes: Response?): String {
        var result = Result(0, "", 0, emptyList())
        return if (resultRes == null || resultRes.result.also {
                result = it
            } == null) "null" else ToolUtil.formatWithUSLocal(
            "Common.Result{errcode: %d, errmsg: %s, magic: %d}",
            result.errcode, result.errmsg, result.magic
        )
    }

    // 编译期间插入具体实现的代码，不要修改此方法, 本地构建使用反射
    @JvmStatic
    private fun <T : GeneratedMessageLite<T, *>> _parseBytes(
        defaultInstance: T,
        data: ByteArray
    ): T? {
        return invokeParse(defaultInstance, data)
    }

    private fun <T : GeneratedMessageLite<T, *>> invokeParse(
        defaultInstance: T,
        data: ByteArray
    ): T? {
        var result: T? = null
        try {
            val testClass: Class<*> = defaultInstance::class.java
            val parseFrom = testClass.getMethod(
                "parseFrom", ByteArray::class.java
            )
            @Suppress("UNCHECKED_CAST")
            result = parseFrom.invoke(null, data) as T?
        } catch (e: Exception) {
            sLogger.logE(TAG, "", e)
        }
        return result
    }

    @JvmStatic
    fun <T : AndroidMessage<T, *>?> parseBytes(
        adapter: ProtoAdapter<T>,
        data: ByteArray
    ): T? {
        try {
            return adapter.decode(data)
        } catch (e: Exception) {
            sLogger.logE(TAG, "", e)
        }
        return null
    }

    fun getActualTypeArgument(obj: Any, clz: Class<*>): Type? {
        var targetType: ParameterizedType? = null
        val objClz: Class<*> = obj.javaClass
        for (type in objClz.genericInterfaces) {
            if (type is ParameterizedType &&
                clz.isAssignableFrom(type.rawType as Class<*>)
            ) {
                targetType = type
            }
        }
        if (targetType == null) {
            val type = objClz.genericSuperclass
            if (type is ParameterizedType &&
                clz.isAssignableFrom(type.rawType as Class<*>)
            ) {
                targetType = type
            }
        }
        if (targetType != null) {
            if (targetType.actualTypeArguments.isNotEmpty()) {
                return targetType.actualTypeArguments[0]
            }
        }
        return null
    }

    @Synchronized
    fun uniqueSeq(): Long {
        return UniqueSeq.uniqueSeq()
    }
}