package com.joyy.hagorpc.internal

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.net.ConnectivityManager
import com.joyy.hagorpc.INetworkChangeListener

/**
 * rpc默认网络状态监听类
 * Created by wjh on 2021/10/28
 */
class RPCNetworkObserver() : BroadcastReceiver() {

    // 当前网络是否可用
    @Volatile
    private var mNetAvailable: Boolean? = null
        set(value) {
            if (field != value) {
                field = value

                synchronized(this) {
                    mListenerList.forEach {
                        it.onChange(field ?: false)
                    }
                }
            }
        }

    private val mListenerList = mutableListOf<INetworkChangeListener>()

    fun hasListener(): Boolean {
        return synchronized(this) {
            mListenerList.isNotEmpty()
        }
    }

    fun addListener(listener: INetworkChangeListener) {
        synchronized(this) {
            if (!mListenerList.contains(listener)) {
                mListenerList.add(listener)
            }
        }
    }

    fun removeListener(listener: INetworkChangeListener) {
        synchronized(this) {
            mListenerList.remove(listener)
        }
    }

    fun isNetAvailable(context: Context?): Boolean {
        var netValid = mNetAvailable
        if (netValid == null) {
            netValid = innerNetAvailable(context)
        }
        return netValid
    }

    private fun innerNetAvailable(context: Context?): Boolean {
        val connectivityManager =
            context?.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager
        val networkInfo = connectivityManager?.activeNetworkInfo
        return if (networkInfo != null) {
            networkInfo.isConnected ||
                    (networkInfo.isAvailable && networkInfo.isConnectedOrConnecting)
        } else {
            false
        }
    }

    fun observe(context: Context?) {
        val filter = IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION)
        context?.registerReceiver(this, filter)
        mNetAvailable = isNetAvailable(context)
    }

    fun cancel(context: Context?) {
        context?.unregisterReceiver(this)
        mNetAvailable = null
    }

    override fun onReceive(context: Context?, intent: Intent?) {
        val netValid = innerNetAvailable(context)
        if (mNetAvailable != netValid) {
            mNetAvailable = netValid

            synchronized(this) {
                mListenerList.forEach {
                    it.onChange(isNetAvailable(context))
                }
            }
        }
    }
}