package com.joyy.hagorpc.internal

import android.text.TextUtils
import android.util.Log
import com.joyy.hagorpc.INetworkClient
import com.joyy.hagorpc.IRPCInternalContext
import com.joyy.hagorpc.birdge.HttpClientBridge
import java.net.SocketTimeoutException
import java.net.UnknownHostException
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.TimeUnit

/**
 * rpc 短连接请求处理类
 * Created by wjh on 2021/12/20
 */
internal class RPCHttpSender(private val mContext: IRPCInternalContext) {

    companion object {
        private const val TAG = "RPCHttpSender"
    }

    private val mUrlEncodeCache = ConcurrentHashMap<String, String>()

    private fun urlEncode(content: String): String {
        if (TextUtils.isEmpty(content)) {
            return ""
        }
        if (mUrlEncodeCache.containsKey(content)) {
            return mUrlEncodeCache[content] ?: ""
        }
        val encode: String = RPCProtoHelper.urlEncoderEncode(content)
        if (!TextUtils.isEmpty(encode)) {
            mUrlEncodeCache[content] = encode
            return encode
        }
        return ""
    }

    /**
     * 发送请求
     */
    fun send(
        httpUrl: String?,
        method: HttpClientBridge.HttpMethod,
        requestHeader: Map<String, String>,
        payload: ByteArray?,
        waitTimeout: Long,
        onResponse: (responseBytes: ByteArray, responseLength: Long, responseHeader: Map<String, List<String>>) -> Unit,
        onError: (canRetry: Boolean, fromTimeout: Boolean, code: Int, msg: String, responseHeader: Map<String, List<String>>) -> Unit
    ): Boolean {
        try {
            if (httpUrl.isNullOrEmpty()) {
                throw UnknownHostException("http url is null")
            }
            val header = mutableMapOf<String, String>()
            header.putAll(requestHeader)
            val requestBytes = payload ?: ByteArray(0)
            mContext.getLogger().logD(
                TAG,
                "sendHttp begin url: ${httpUrl}, \n" +
                        "header: $header"
            )
            val requestBuilder = INetworkClient.Request.Builder()
                .url(httpUrl)
                .addHeader(header)

            when (method) {
                HttpClientBridge.HttpMethod.GET ->
                    requestBuilder.get()
                HttpClientBridge.HttpMethod.POST ->
                    requestBuilder.post(
                        "application/proto",
                        requestBytes
                    )
            }

            if (waitTimeout > 0) {
                requestBuilder.waitTimeout(waitTimeout, TimeUnit.MILLISECONDS)
            }
            mContext.getNetworkClient()
                .newCall(requestBuilder.build())
                .enqueue(object : INetworkClient.ICallback {
                    override fun onResponse(
                        response: INetworkClient.IResponse
                    ) {
                        try {
                            if (mContext.getLogger().isDebuggable()) {
                                mContext.getLogger().logD(
                                    TAG, "sendHttp onResponse\n" +
                                            "send result: ${response.code()}, \n" +
                                            "header: ${
                                                response.headersToString()
                                            }, \nurl: $httpUrl"
                                )
                            }
                            val responseHeader = response.responseHeaderFields()
                            if (response.isSuccessful()) {
                                val responseBytes = response.bytes()
                                val responseLength = responseBytes.size.toLong()

                                onResponse(
                                    responseBytes,
                                    responseLength,
                                    responseHeader
                                )
                            } else {
                                onError(
                                    false,
                                    false,
                                    response.code(),
                                    "response fail, error code: ${response.code()}",
                                    responseHeader
                                )
                            }
                        } finally {
                            Log.d(TAG, "onResponse close body, release resource")
                            response.close()
                        }
                    }

                    override fun onFailure(throwable: Throwable?) {
                        val canRetry = checkCanRetry(throwable)
                        onError(
                            canRetry,
                            throwable is SocketTimeoutException,
                            mContext.getNetworkClient().getErrorCodeByExc(null, throwable),
                            throwable?.message ?: "",
                            emptyMap()
                        )
                    }
                })
            return true
        } catch (ex: Exception) {
            val canRetry = checkCanRetry(ex)
            onError(
                canRetry,
                ex is SocketTimeoutException,
                mContext.getNetworkClient().getErrorCodeByExc(null, ex),
                ex.message ?: "",
                emptyMap()
            )
        }
        return false
    }

    private fun checkCanRetry(throwable: Throwable?): Boolean {
        return RPCProtoHelper.isNetworkException(throwable)
    }
}