package com.joyy.hagorpc.internal

import com.joyy.hagorpc.RPCCommonNotice
import com.joyy.hagorpc.IRPCNotify
import com.joyy.hagorpc.IRegisterId
import com.joyy.hagorpc.NotifyBundle
import com.joyy.hagorpc.RPCNotifyExtra
import com.joyy.hagorpc.RPCPacketV2
import com.joyy.hagorpc.birdge.LoggerBridge
import com.joyy.hagorpc.birdge.RPCPacketBridge

/**
 * 广播分发器
 * Created by wjh on 2021/10/25
 */
internal class RPCDispatcherImpl : AbsRPCDispatcher() {

    override fun logTag(): String {
        return "RpcDispatcher"
    }

    override fun findAction(
        bundle: NotifyBundle,
        observerMap: Map<IRegisterId, List<IRPCNotify>>
    ): List<NotifyAction> {

        val actionList = mutableListOf<NotifyAction>()
        try {
            var action = innerFindAction(
                bundle.protocol,
                bundle.protocol.header.sname,
                bundle.protocol.uri.toLong(),
                observerMap
            )
            if (action != null) {
                action.payload = bundle.payload
                actionList.add(action)
            }

            val commonNotice = getCommonNotice(bundle.protocol, bundle.payload)
            if (commonNotice != null) {
                action = innerFindAction(
                    bundle.protocol,
                    commonNotice.sname,
                    commonNotice.uri,
                    observerMap
                )
                if (action != null) {
                    action.payload = commonNotice.payload
                    actionList.add(action)
                }
            }

            if (actionList.isNotEmpty()) {
                LoggerBridge.logD(
                    logTag(),
                    "find action bundle: $bundle, actionList: $actionList"
                )
            } else {
                LoggerBridge.logW(logTag(), "find action list empty, bundle: $bundle")
            }
        } catch (ex: Exception) {
            LoggerBridge.logE(logTag(), "find action error, bundle: $bundle")
        }
        return actionList
    }

    private fun getCommonNotice(protocol: RPCPacketV2, payload: ByteArray): RPCCommonNotice? {
        var result: RPCCommonNotice? = null
        try {
            if (protocol.header.sname == RPCServiceDefine.YPUSH) {
                result = RPCPacketBridge.fromCommonNotice(payload)
            }
        } catch (ex: Exception) {
            LoggerBridge.logE(
                logTag(),
                "get common notice header null"
            )
        }
        return result
    }

    private fun innerFindAction(
        protocol: RPCPacketV2,
        service: String,
        uri: Long,
        observerMap: Map<IRegisterId, List<IRPCNotify>>
    ): NotifyAction? {
        if (observerMap.isEmpty()) {
            return null
        }
        val uiList = mutableListOf<IRPCNotify>()
        val threadList = mutableListOf<IRPCNotify>()
        for ((registerId, observers) in observerMap) {
            if (registerId.isMatch(service, uri)) {
                observers.forEach { observer ->
                    if (observer.notifyOnUI()) {
                        uiList.add(observer)
                    } else {
                        threadList.add(observer)
                    }
                }
            }
        }

        if (uiList.isNotEmpty() || threadList.isNotEmpty()) {
            return NotifyAction.obtain().apply {
                this.extra = RPCNotifyExtra(
                    service, protocol.header.method,
                    uri, protocol.header
                )
                this.uiObserver = uiList
                this.threadObserver = threadList
            }
        }
        return null
    }
}