package com.joyy.hagorpc.internal

import android.os.Handler
import android.os.HandlerThread
import android.os.Looper
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.ExecutorService
import java.util.concurrent.SynchronousQueue
import java.util.concurrent.ThreadFactory
import java.util.concurrent.ThreadPoolExecutor
import java.util.concurrent.TimeUnit

/**
 * rpc内部默认线程处理
 * Created by wjh on 2021/10/28
 */
object RPCDefaultHandler {

    private val mDelayTaskMap = ConcurrentHashMap<Runnable, Runnable>()

    private val mUiHandler: Handler by lazy {
        Handler(Looper.getMainLooper())
    }

    private val mQueueHandler: Handler by lazy {
        val thread = HandlerThread("HAGO_RPC_queue_thread")
        thread.start()
        Handler(thread.looper)
    }

    private val mExecutorService: ExecutorService by lazy {
        ThreadPoolExecutor(
            0 /* corePoolSize */,
            Int.MAX_VALUE, 60L /* keepAliveTime */,
            TimeUnit.SECONDS,
            SynchronousQueue(),
            ThreadFactory { runnable: Runnable? ->
                val result = Thread(runnable, "HAGO_RPC_ThreadPool")
                result.isDaemon = true
                result
            })
    }

    fun getThreadPool(): ExecutorService {
        return mExecutorService
    }

    fun post(task: Runnable) {
        mUiHandler.post(task)
    }

    fun post(task: Runnable, delay: Long) {
        if (delay <= 0) {
            post(task)
        } else {
            mUiHandler.postDelayed(task, delay)
        }
    }

    fun postAsync(task: Runnable) {
        mExecutorService.execute(task)
    }

    fun postAsync(task: Runnable, delay: Long) {
        if (delay <= 0) {
            postAsync(task)
        } else {
            val delayTask = Runnable {
                task.run()
            }
            mDelayTaskMap[task] = delayTask
            post(delayTask, delay)
        }
    }

    fun postQueue(task: Runnable) {
        mQueueHandler.post(task)
    }

    fun postQueue(task: Runnable, delay: Long) {
        if (delay <= 0) {
            postQueue(task)
        } else {
            mQueueHandler.postDelayed(task, delay)
        }
    }

    fun remove(task: Runnable) {
        mUiHandler.removeCallbacks(task)
    }

    fun removeQueue(task: Runnable) {
        mQueueHandler.removeCallbacks(task)
    }

    fun removeAsync(task: Runnable) {
        val delayTask = mDelayTaskMap.remove(task)
        if (delayTask != null) {
            mUiHandler.removeCallbacks(delayTask)
        }
    }
}