package com.joyy.hagorpc.internal

import com.joyy.hagorpc.IFrequencyExecutor
import com.joyy.hagorpc.IMemoryChangeListener
import com.joyy.hagorpc.INetworkChangeListener
import com.joyy.hagorpc.INotifyFrequencyDelegate
import com.joyy.hagorpc.IRPCDataProvider.DispatchCondition
import com.joyy.hagorpc.IRPCDataProvider.DispatchResult
import com.joyy.hagorpc.IRPCDataProvider.HeartbeatInfo
import com.joyy.hagorpc.IRPCEventListener
import com.joyy.hagorpc.IRPCMemoryDelegate
import com.joyy.hagorpc.IRPCNotify
import com.joyy.hagorpc.MSGTYPE
import com.joyy.hagorpc.NotifyBundle
import com.joyy.hagorpc.RPCCallException
import com.joyy.hagorpc.RPCCallRequest
import com.joyy.hagorpc.RPCCallResponse
import com.joyy.hagorpc.RPCHeader
import com.joyy.hagorpc.RPCPacketV2
import com.joyy.hagorpc.WsCode
import com.joyy.hagorpc.WsStatus
import com.joyy.hagorpc.birdge.ExecutorBridge
import com.joyy.hagorpc.birdge.LoggerBridge
import com.joyy.hagorpc.birdge.NetworkBridge
import com.joyy.hagorpc.birdge.RunnableBridge
import com.joyy.hagorpc.birdge.SystemBridge
import com.joyy.hagorpc.birdge.SystemClockBridge
import com.joyy.hagorpc.birdge.WsClientBridge
import com.joyy.hagorpc.impl.DefaultEventListener
import kotlinx.atomicfu.atomic
import kotlin.math.abs
import kotlin.random.Random

object RPCCore : INetworkChangeListener {
    // 业务层主动关闭 web socket, 区分异常关闭
    private const val MYSELF_CLOSE_REASON = "close by myself"
    private const val TAG = "RPCCore"
    private const val BACKGROUND_MAX_TIME_RECONNECT: Long = 180000 // 3min

    // 重连间隔进行增量随机，防雪崩
    private val intervals = longArrayOf(500, 1000, 3000, 5000, 8000, 12000)

    interface RPCCallback {

        /**
         * 是否期望使用短连http请求
         */
        fun expectHttpRequest(): Boolean = false

        /**
         * 请求是否需要token
         */
        fun needToken(): Boolean = true

        /**
         * 默认超时
         */
        fun getTimeout(): Long = RPCConst.DEFAULT_TASK_TIMEOUT

        /**
         * 请求发送前回调
         */
        fun onRequest(req: RPCCallRequest) = Unit

        /**
         * RPC 请求成功回调
         */
        fun onSuccess(
            response: RPCCallResponse
        )

        /**
         * RPC 请求失败回调
         * @return 是否需要重试
         */
        fun onError(
            error: RPCCallException
        ): Boolean
    }

    interface WsCallback {
        /**
         * web socket 连接成功回调
         */
        fun onConnectSucceed(wsUrl: String)

        /**
         * 重连次数达到最大重连次数回调，不会再进行重连尝试
         */
        fun onReConnectMax(wsUrl: String)

        /**
         * 正在连接中
         */
        fun onConnecting(wsUrl: String)


        /**
         * 收到 ByteString 类型信息回调
         *
         */
        fun onResponse(wsUrl: String, bytes: ByteArray)

        /**
         * 每次失败都会回调
         */
        fun onError(wsUrl: String, code: Int, reason: String)

        /**
         * 主动调用关闭web socket 成功回调，返回码 4000-4100 不需要重连
         */
        fun onDisconnect(wsUrl: String, code: Int, reason: String)
    }

    // 是否触发建立长连接
    private var mCallConnect = atomic(false)
    private val mWsCallbackProxy = WsCallbackProxy()
    var mWebSocket: WsClientBridge.WebSocket? = null
    private var mWebSocketId: Long = -1
    private var mStatus = WsStatus.CONNECT_NONE
    var status: WsStatus
        get() = mStatus
        private set(status) {
            if (mStatus !== status) {
                mStatus = status
                when (status) {
                    WsStatus.CONNECTING -> {
                        LoggerBridge.logI(TAG, "长连接连接中...")
                    }
                    WsStatus.CONNECT_FAIL -> {
                        LoggerBridge.logI(TAG, "长连接连接失败")
                    }
                    WsStatus.CONNECT_SUCCESS -> {
                        LoggerBridge.logI(TAG, "长连接连接成功")
                    }
                    WsStatus.CONNECT_NONE -> {
                        LoggerBridge.logI(TAG, "长连接状态重置")
                    }
                    else -> {}
                }
            }
        }
    private lateinit var mEnableBackgroundReconnect: (wsUrl: String) -> Boolean
    lateinit var mToken: () -> String
    lateinit var mUid: () -> String
    private lateinit var mWsUrl: () -> String
    private lateinit var mHttpUrl: () -> String
    lateinit var mUseGZip: (wsUrl: String) -> Boolean
    lateinit var mDeviceId: String
    lateinit var mLanguage: String
    lateinit var mCountry: String
    private lateinit var mUseReliableBroadcast: (wsUrl: String) -> Boolean
    internal lateinit var mEventListener: IRPCEventListener
    internal lateinit var mHeartbeatInfo: (from: Int?) -> HeartbeatInfo
    internal lateinit var mMemoryDelegate: IRPCMemoryDelegate
    internal lateinit var mNotifyFrequencyDelegate: INotifyFrequencyDelegate
    internal lateinit var mDispatchRequestToHttp: (condition: DispatchCondition) -> DispatchResult?
    internal lateinit var mCommonHeader: (wsUrl: String, forWs: Boolean) -> Map<String, String>
    private lateinit var mDisableReconnect: (wsUrl: String, errCode: Int) -> Boolean
    var mRequestMaxRetry: Int = RPCConst.REQUEST_RETRY_MAX
    private var mConnectRetryMax: Int = RPCConst.CONNECT_RETRY_MAX
    private var mConnectTimeout: Long = RPCConst.CONNECT_TIMEOUT
    private var mPingInterval: Long = RPCConst.PING_INTERVAL
    var mResendIfConnected = RPCConst.RESEND_IF_CONNECTED

    // 是否在后台
    val mIsBackground = atomic(false)

    // 网络状况
    private val mNetValid = atomic(false)

    // 没有Open之前都不需要重连
    private val mOpen = atomic(false)
    private val mBackgroundTime = atomic(-1L) // 切后台时间戳，前台为-1

    // seq - 因为WSClient 会自动重连复用，所以需要在这里加一个连接序号
    // 某些操作真的的是WSClient 的某一次连接序号
    private val mConnectSeq = atomic(0)

    /**
     * 服务器断连的原因，每次成功建立连接需要重置为0，尝试重连的时候如果发现是服务器主动断连，则不重连
     */
    private val mReasonCode = atomic(0)

    private var mReConnectTask: ReopenTask? = null

    private val mCurReconnectCount = atomic(0)

    // 请求缓存队列
    internal val mRequestMap = atomic(LinkedHashMap<Long, RPCRequest>())

    // 产生唯一的序列号
    private val sSeq = atomic(0)

    fun newSocketSeq(): Int {
        return sSeq.incrementAndGet()
    }

    // 类似于socket的句柄，每一次连接都会有唯一的序列号
    fun identifySeq(): Int {
        return mConnectSeq.value
    }

    private val mOpenSuccessTask = RunnableBridge {
        LoggerBridge.logD(TAG, "connect success, reset connect count")
        mCurReconnectCount.value = 0
    }

    /**
     * @param token 获取 token
     * @param requestMaxRetry 重试最大次数
     * @param enableBackgroundReconnect 是否启用后台重试
     * @param useGZip 是否启用 GZip
     * @param useReliableBroadcast 是否启用可靠广播
     */
    fun config(
        token: () -> String,
        uid: () -> String,
        wsUrl: () -> String,
        httpUrl: () -> String,
        deviceId: String = "",
        language: String = "",
        country: String = "",
        enableBackgroundReconnect: (wsUrl: String) -> Boolean = { true },
        useGZip: (wsUrl: String) -> Boolean = { false },
        useReliableBroadcast: (wsUrl: String) -> Boolean = { false },
        requestMaxRetry: Int = RPCConst.REQUEST_RETRY_MAX,
        connectRetryMax: Int = RPCConst.CONNECT_RETRY_MAX,
        connectTimeout: Long = RPCConst.CONNECT_TIMEOUT,
        pingInterval: Long = RPCConst.PING_INTERVAL,
        eventListener: IRPCEventListener = DefaultEventListener(),
        resendIfConnected: Boolean = RPCConst.RESEND_IF_CONNECTED,
        heartbeatInfo: (from: Int?) -> HeartbeatInfo = {
            HeartbeatInfo(
                useCProxy = true,
                onlineBizMap = null,
                requestTimeout = 10000,
                pongTimeout = 15000 * 3 + 3000,
                playingGame = null
            )
        },
        memoryDelegate: IRPCMemoryDelegate = object : IRPCMemoryDelegate {
            override fun onObserveMemory(listener: IMemoryChangeListener) {
                // Do nothing
            }

            override fun onRemoveObserver(listener: IMemoryChangeListener) {
                // Do nothing
            }
        },
        notifyFrequencyDelegate: INotifyFrequencyDelegate = object : INotifyFrequencyDelegate {
            override fun createExecutor(duration: Long): IFrequencyExecutor =
                object : IFrequencyExecutor {
                    override fun execute(task: RunnableBridge) {
                        task.run()
                    }

                    override fun update(duration: Long) {
                        // Do nothing
                    }

                }

            override fun getIntervalTime(sname: String, uri: Int): Long = 0

            override fun canReplace(sname: String, uri: Int): Boolean = false

            override fun onBatch(sname: String, uri: Int): Int = -1

            override fun isControl(sname: String): Boolean = false

            override fun isControl(sname: String, uri: Int): Boolean = false

            override fun onAfterNotify(
                sname: String,
                uriList: List<Int>,
                lastNotifyTime: Long,
                thisNotifyTime: Long,
                receiveTime: Long
            ) = Unit

            override fun intercept(dataList: List<NotifyBundle>): List<NotifyBundle> = dataList

        },
        dispatchRequestToHttp: (condition: DispatchCondition) -> DispatchResult? = {
            DispatchResult(
                useHttp = it.expectUseHttp,
                useWaitTimeout = false
            )
        },
        commonHeader: (wsUrl: String, forWs: Boolean) -> Map<String, String> = { _, _ -> emptyMap() },
        disableReconnect: (wsUrl: String, errCode: Int) -> Boolean = { _, _ -> false }
    ) {
        mRequestMaxRetry = requestMaxRetry
        mConnectRetryMax = connectRetryMax
        mEnableBackgroundReconnect = enableBackgroundReconnect
        mUseGZip = useGZip
        mUseReliableBroadcast = useReliableBroadcast
        mToken = token
        mUid = uid
        mWsUrl = wsUrl
        mHttpUrl = httpUrl
        mDeviceId = deviceId
        mLanguage = language
        mCountry = country
        mConnectTimeout = connectTimeout
        mPingInterval = pingInterval
        mEventListener = eventListener
        mResendIfConnected = resendIfConnected
        mHeartbeatInfo = heartbeatInfo
        mMemoryDelegate = memoryDelegate
        mNotifyFrequencyDelegate = notifyFrequencyDelegate
        mDispatchRequestToHttp = dispatchRequestToHttp
        mCommonHeader = commonHeader
        mDisableReconnect = disableReconnect

        // 初始化完毕就开始轮询等待队列
        mWsCallbackProxy.triggerTimeoutTask()
    }

    private fun getWsUrl(): String = "${mWsUrl.invoke()}/ikxd_cproxy?token=${mUid.invoke()}"
    private fun getHttpUrl(): String = mHttpUrl.invoke()

    /**
     * 建立长连接
     */
    fun connectWebSocket(
        forceReConnect: Boolean = false,
        callback: WsCallback
    ) {
        val webSocket = mWebSocket
        if (webSocket == null) {
            if (mCallConnect.compareAndSet(expect = false, update = true)) {
                LoggerBridge.logI(TAG, "connectWebSocket new")
                innerConnect(callback)
            }
        } else if (forceReConnect || (webSocket.getWsUrl() != getWsUrl() &&
                    status == WsStatus.CONNECT_FAIL)
        ) {
            LoggerBridge.logI(TAG, "connectWebSocket force")
            // 如果是强制或之前的客户端未建立连接且地址不一致时，触发连接
            innerConnect(callback)
        }
    }

    fun triggerHeartbeat(from: Int?) {
        mWsCallbackProxy.startHeartBeat(getWsUrl(), from)
    }

    /**
     * 断开长连接
     */
    fun disconnectWebSocket() {
        ExecutorBridge.getAsyncExecutor().execute {
            LoggerBridge.logI(TAG, "disconnect url: ${getWsUrl()}")
            destroyWsClient()
        }
    }

    fun register(id: RegisterId, notify: IRPCNotify) {
        LoggerBridge.logI(
            TAG,
            "register service: ${id.service}, uri: ${id.uri}, fullMatch: ${id.fullMatch}"
        )
        mWsCallbackProxy.register(id, notify)
    }

    fun unregister(id: RegisterId, notify: IRPCNotify) {
        LoggerBridge.logI(
            TAG,
            "unregister service: ${id.service}, uri: ${id.uri}, fullMatch: ${id.fullMatch}"
        )
        mWsCallbackProxy.unregister(id, notify)
    }

    fun getHeaderBuilder(
        sName: String,
        v2: Boolean
    ): RPCHeader.Builder {

        val builder = RPCHeader.newBuilder()
            .setCode(0)
            .setSeqid(UniqueSeq.uniqueSeq())
            .setSname(sName)
        var byteString: ByteArray? = null
        val proxyRouting = RPCLocalPersistence.getCProxyRouting(sName)
        if (proxyRouting.isNotEmpty()) {
            byteString = proxyRouting.encodeToByteArray()
        }
        byteString?.let {
            builder.setRoutingKey(byteString)
        }
        builder.setBackGround(mIsBackground.value)
        if (v2) {
            builder
                .setMsgType(MSGTYPE.MSGTYPE_REQ)
                .setVersion("0.0.0")
        }
        return builder
    }

    /**
     * 发送 RPC 请求
     * @param httpUrl WebSocket 服务器地址
     * @param packet 数据包
     * @param checkByHttp 是否需要检查 Http
     * @param callback 回调
     * @return 是否发送成功
     */
    fun send(
        packet: RPCPacketV2,
        checkByHttp: Boolean = true,
        routingKey: String? = null,
        callback: RPCCallback
    ): Boolean {
        val rpcRequest = RPCRequest(
            wsUrl = getWsUrl(),
            httpUrl = getHttpUrl(),
            packet = packet,
            routingKey = routingKey,
            callback = callback,
            checkByHttp = checkByHttp
        )
        rpcRequest.updateSequenceId(packet.header.seqid)
        return rpcRequest.send()
    }

    fun changeToForeground() {
        LoggerBridge.logI(TAG, "前后台发生变化: 前台")
        mIsBackground.value = false
        if (mOpen.value) {
            mBackgroundTime.value = -1
            if (mNetValid.value && !checkConnected()) {
                createReopenTask("发起重连：前后台切换：切到前台", 0)
            }
            //按照与后台的约定，后台切到前台时，立即调用一次心跳；
            mWsCallbackProxy.startHeartBeat(getWsUrl())
        }
        innerEventForForegroundChange(true)
    }

    fun changeToBackground() {
        LoggerBridge.logI(TAG, "前后台发生变化: 后台")
        mBackgroundTime.value = SystemClockBridge.elapsedRealtime()
        innerEventForForegroundChange(false)
    }

    private fun innerEventForForegroundChange(isForeground: Boolean) {
        val cur = SystemBridge.currentTimeMillis()
        val requestMap = mRequestMap.value
        for ((_, request) in requestMap) {
            val timeCheck = cur - request.sendTime
            request.trace +=
                (if (isForeground) "_fore" else "_back") + "($timeCheck)"
        }
        val content = if (isForeground) {
            "_fore"
        } else {
            "_back"
        } + "(${(SystemClockBridge.uptimeMillis() - mWsCallbackProxy.mLastTimeoutCheck)})"
        LoggerBridge.logD(TAG, "stat foreground change: $content")
        mWsCallbackProxy.mLastTimeoutCheckTrace.append(content)
    }

    /**
     * 检测是否已连接
     */
    fun checkConnected(): Boolean {
        if (mWebSocket == null) {
            LoggerBridge.logE(TAG, "web socket had not create")
            return false
        }
        if (status !== WsStatus.CONNECT_SUCCESS) {
            LoggerBridge.logE(TAG, "web socket had not connected")
            return false
        }
        return true
    }

    fun createReopenTask(log: String, delay: Long) {
        cancelReopenTask()
        val reConnectTask = ReopenTask(log)
        mReConnectTask = reConnectTask
        if (delay <= 0) {
            ExecutorBridge.getAsyncExecutor().execute(reConnectTask)
        } else {
            ExecutorBridge.getAsyncExecutor().executeDelay(reConnectTask, delay)
        }
    }

    override fun onChange(isNetAvailable: Boolean) {
        if (mNetValid.compareAndSet(!isNetAvailable, isNetAvailable)) {
            LoggerBridge.logD(TAG, "network change: $isNetAvailable")

            // 网络变更的相关处理
            if (!mOpen.value) {
                // 没有打开的客户端不需要处理这个变更事件
                return
            }
            if (isNetAvailable) {
                // tryReconnect right now
                LoggerBridge.logI(TAG, "the network is connected, tryReconnect right now")
                mCurReconnectCount.value = 0 //当网络由未连接变成已连接，重置重试次数，加快连接；
                checkReopen(0, "发起重连：网络连接状态变成已连接")
            } else {
                LoggerBridge.logI(TAG, "the network  disconnected")
            }
        }
    }

    /**
     * 获取当前请求队列的个数pair，first，ws个数，second，http个数
     * 如果current不为空，则要减去当前current所属
     */
    internal fun getRequestQueueSize(current: RPCRequest?): Pair<Int, Int> {
        var wsCount = 0
        var httpCount = 0
        val requestMap = mRequestMap.value
        for ((_, request) in requestMap) {
            // 自己不算
            if (request == current) {
                continue
            }
            if (!request.isHttpRequesting) {
                wsCount++
            } else {
                httpCount++
            }
        }
        return Pair(wsCount, httpCount)
    }

    private fun handleError(code: Int, exception: Throwable?) {
        // 清理存根
        close(false, "连接服务器失败，关闭后重试")
        status = WsStatus.CONNECT_FAIL
        checkReopen(reconnectInterval(), "发起重连：服务器连接失败")
        mWsCallbackProxy.onError(
            getWsUrl(),
            code, exception?.toString() ?: ""
        )
    }

    private fun handleDisconnect(code: Int, reason: String) {
        // 清理存根
        close(false, "服务器主动关闭")
        status = WsStatus.CONNECT_FAIL
        mWsCallbackProxy.onDisconnect(getWsUrl(), code, reason)
    }

    private fun close(resetState: Boolean, reasonLog: String) {
        val wsUrl = getWsUrl()
        LoggerBridge.logI(
            TAG,
            "close resetState: $resetState, reasonLog: $reasonLog, mCurStatus: $status, mUri: $wsUrl",
        )

        // 先赋值给别人，再把存根清理掉，最后再关闭socket
        val socket = mWebSocket
        mWebSocket = null
        mWebSocketId = -1
        if (socket != null) {
            LoggerBridge.logI(
                TAG,
                "关闭连接,原因-$reasonLog 连接地址-$wsUrl 关闭前状态-$status"
            )
            if (resetState) {
                status = WsStatus.CONNECT_NONE
            }
            socket.close(WsCode.MYSELF_CLOSE_CODE, MYSELF_CLOSE_REASON)
        }
    }

    private fun open(callback: WsCallback) {
        if (mOpen.compareAndSet(expect = false, update = true)) {
            listenNetwork()
            val netValid = NetworkBridge.getNetworkDelegate().isNetworkAvailable()
            mNetValid.value = netValid
            mReasonCode.value = 0
            mWsCallbackProxy.callback = callback
            // 网络可用打开连接
            if (netValid) {
                LoggerBridge.logI(TAG, "打开")
                connect("打开")
            } else {
                LoggerBridge.logI(TAG, "打开：失败，网络未连接")
            }
        }
    }

    private fun connect(connectLogType: String) {
        val socketSeq = newSocketSeq()
        val wsUrl = getWsUrl()
        mConnectSeq.value = socketSeq
        val webSocketId = UniqueSeq.uniqueSeq()
        LoggerBridge.logI(
            TAG, "开始连接：$connectLogType, seq: $socketSeq, 创建连接url: $wsUrl, webSocketId: $webSocketId"
        )
        val header = mutableMapOf<String, String>()
        mCommonHeader.invoke(wsUrl, true).forEach { (key, value) ->
            if (value.isNotEmpty()) {
                header[key] = value;
            }
        };
        if (mUseGZip.invoke(wsUrl)) {
            header["Content-Encoding"] = "gzip"
        }
        if (mUseReliableBroadcast.invoke(wsUrl)) {
            header["X-Last-Seqid"] = RPCLocalPersistence.getBroadcastSeq(
                mUid.invoke(), 0
            ).toString() + ""
        }
        val authToken = mToken.invoke()
        if (authToken.isNotEmpty()) {
            header[RPCConst.X_AUTH_TOKEN] = authToken
        }
        if (mDeviceId.isNotEmpty()) {
            header[RPCConst.X_DEVICE_ID] = mDeviceId
        }
        if (mCountry.isNotEmpty()) {
            header[RPCConst.X_SIM_CI_SO] = mCountry
        }
        if (mLanguage.isNotEmpty()) {
            if (mCountry.isNotEmpty()) {
                header[RPCConst.X_LANG] = "${mLanguage}_${mCountry.uppercase()}"
            } else {
                header[RPCConst.X_LANG] = mLanguage
            }
        }

        mWebSocket =
            WsClientBridge.newWebSocket(wsUrl, header, object : WsClientBridge.WebSocketListener {

                override fun onOpen() {
                    LoggerBridge.logI(
                        TAG,
                        "onOpen url: $wsUrl, webSocketId: $webSocketId, mWebSocketId: $mWebSocketId"
                    )
                    if (webSocketId != mWebSocketId) {
                        return
                    }
                    cancelReopenTask()
                    status = WsStatus.CONNECT_SUCCESS
                    mWsCallbackProxy.onConnectSucceed(wsUrl)
                    ExecutorBridge.getMainExecutor().executeDelay(mOpenSuccessTask, 2000)
                }

                override fun onMessage(bytes: ByteArray) {
                    LoggerBridge.logD(
                        TAG,
                        "onMessage url: $wsUrl, webSocketId: $webSocketId, mWebSocketId: $mWebSocketId"
                    )
                    if (webSocketId != mWebSocketId) {
                        return
                    }
                    mWsCallbackProxy.onResponse(wsUrl, bytes)
                }

                override fun onClose(
                    code: Int,
                    reason: String
                ) {
                    LoggerBridge.logI(
                        TAG,
                        "onClose url: $wsUrl, code: $code, reason: $reason, webSocketId: $webSocketId, mWebSocketId: $mWebSocketId"
                    )
                    if (webSocketId != mWebSocketId) {
                        return
                    }
                    ExecutorBridge.getMainExecutor().cancel(mOpenSuccessTask)
                    // NB!
                    // https://segmentfault.com/a/1190000007587248
                    // 4000–4999 可以由应用使用
                    // 4000-4100 服务的返回该范围，不需要重连
                    // 约定服务端主动踢掉返回错误码: 4000
                    // 第二个连接过来的时候，会把第一个连接干掉

                    // 因为4000-4999都是业务定义的，所以改为这个范围的错误码都不重连，直接关闭
                    if (code == WsCode.MYSELF_CLOSE_CODE || code >= 4000 && code <= 5000 || mDisableReconnect.invoke(
                            wsUrl,
                            code
                        )
                    ) {
                        mReasonCode.value = code
                        handleDisconnect(code, reason)
                    } else {
                        handleError(code, RuntimeException("receive close code: $code"))
                    }
                }

                override fun onError(
                    errorMessage: String
                ) {
                    LoggerBridge.logI(
                        TAG,
                        "onFailure url: $wsUrl, throwable: $errorMessage, webSocketId: $webSocketId, mWebSocketId: $mWebSocketId",
                    )
                    if (webSocketId != mWebSocketId) {
                        return
                    }
                    ExecutorBridge.getMainExecutor().cancel(mOpenSuccessTask)
                    handleError(-1, Exception(errorMessage))//(getFailureCode(throwable), throwable)
                }
            })
        mWebSocketId = webSocketId
        status = WsStatus.CONNECTING
        mWsCallbackProxy.onConnecting(wsUrl)
        val webSocket = mWebSocket
        if (webSocket == null) {
            status = WsStatus.CONNECT_FAIL
            checkReopen(reconnectInterval(), "发起重连：创建连接异常")
            LoggerBridge.logI(TAG, "url: $wsUrl")
            mWebSocketId = -1
        }
    }

    private fun reconnectInterval(): Long {
        var spends: Long = 500
        var randomSpends: Long = 0
        var maxConnect = mConnectRetryMax
        if (maxConnect <= 0) {
            maxConnect = intervals.size
        }
        val reconnect = mCurReconnectCount.value
        val rc = reconnect % maxConnect
        if (intervals.size > rc) {
            spends = intervals[rc]
            randomSpends =
                (Random(SystemBridge.currentTimeMillis()).nextDouble() * 500 * reconnect).toLong()
        }
        LoggerBridge.logD(
            TAG,
            "reconnectInterval spend: $spends, random: $randomSpends, " +
                    "maxConnect: $maxConnect, reconnect: $reconnect, rc: $rc"
        )
        return spends + randomSpends
    }

    /**
     * 判断是否可重连
     * @param delay
     * @param tryLog
     */
    private fun checkReopen(delay: Long, tryLog: String) {
        val maxRetry = mConnectRetryMax
        val wsUrl = getWsUrl()
        if (maxRetry == -1 || mCurReconnectCount.value < maxRetry) {
            LoggerBridge.logI(TAG, "tryReconnect url: $wsUrl, delay: $delay")
            createReopenTask(tryLog, delay)
        } else if (maxRetry > 0) {
            // 重连次数超过最大次数
            mWsCallbackProxy.onReConnectMax(wsUrl)
            LoggerBridge.logI(TAG, "connect failed after try max times: $mCurReconnectCount")
        } else {
            LoggerBridge.logI(TAG, "tryReconnect failed, the status wrong $mStatus")
        }
    }

    private fun innerConnect(callback: WsCallback) {
        ExecutorBridge.getAsyncExecutor().execute {
            createWsClient(callback)
            mCallConnect.compareAndSet(expect = true, update = false)
        }
    }

    private fun createWsClient(callback: WsCallback) {
        val recordBegin = SystemBridge.currentTimeMillis()
        val wsUrl = getWsUrl()
        if (wsUrl.isNotEmpty()) {
            destroyWsClient()
            open(callback)
            LoggerBridge.logI(
                TAG,
                "create client duration: ${SystemBridge.currentTimeMillis() - recordBegin}"
            )
        }
    }

    /**
     * 根据长连接异常获取对应的状态码
     */
    private fun getFailureCode(throwable: Throwable): Int {
        val throwableName = throwable::class.simpleName ?: ""
        return if (throwableName.contains("UnknownHostException", ignoreCase = true) ||
            throwableName.contains("NoRouteToHostException", ignoreCase = true) ||
            throwableName.contains("ConnectException", ignoreCase = true) ||
            throwableName.contains("SocketTimeoutException", ignoreCase = true) ||
            throwableName.contains("SSLHandshakeException", ignoreCase = true)
        ) {
            WsCode.NETWORK_FAILURE_CODE
        } else WsCode.UNKNOWN_CODE
    }

    private fun destroy() {
        if (mOpen.compareAndSet(expect = true, update = false)) {
            removeNetworkListen()
            cancelReopenTask()
            close(true, "销毁该条长连接")
            // 关闭的时候要马上回调到上层来处理（不用等websocket回调），主要是清理等待队列来回收资源
            mWsCallbackProxy.onDisconnect(getWsUrl(), WsCode.MYSELF_CLOSE_CODE, MYSELF_CLOSE_REASON)
            mWsCallbackProxy.callback = null
        }
    }

    private fun destroyWsClient() {
        val recordBegin = SystemBridge.currentTimeMillis()

        destroy()

        LoggerBridge.logD(
            TAG,
            "destroy client duration: ${SystemBridge.currentTimeMillis() - recordBegin}"
        )

    }

    private fun listenNetwork() {
        NetworkBridge.getNetworkDelegate().onObserveNetwork(this)
    }

    private fun removeNetworkListen() {
        NetworkBridge.getNetworkDelegate().onRemoveObserver(this)
    }

    private class ReopenTask(private val tryLog: String) : RunnableBridge {
        override fun run() {
            innerHandleReopen(tryLog)
        }
    }

    private fun cancelReopenTask() {
        val reConnectTask = mReConnectTask
        if (reConnectTask != null) {
            ExecutorBridge.getAsyncExecutor().cancel(reConnectTask)
            mReConnectTask = null
        }
    }

    /**
     * 内部处理重新打开行为
     */
    private fun innerHandleReopen(tryLog: String) {
        if (mOpen.value) {
            val reasonCode = mReasonCode.value
            if (reasonCode != 0) {
                LoggerBridge.logI(
                    TAG,
                    "disconnectReason is $reasonCode ,stop reconnecting"
                )
                return
            }
            if (!mNetValid.value) {
                LoggerBridge.logI(
                    TAG,
                    "as the network is invalid, stop reconnecting"
                )
                return
            }
            val cur = SystemClockBridge.elapsedRealtime()
            var interval: Long = 0
            val backgroundFlag = mBackgroundTime.value
            //不开启后台强制重连的话：在后台一段时间后，断开连接不会继续尝试重连，等到切到前台才进行重连；
            if (backgroundFlag > 0 && !mEnableBackgroundReconnect.invoke(getWsUrl()) && abs(cur - backgroundFlag)
                    .also { it -> interval = it } > BACKGROUND_MAX_TIME_RECONNECT
            ) {
                LoggerBridge.logI(
                    TAG,
                    "tryReconnect mBackgroundTime: ${mBackgroundTime.value}, stay background: $interval, exceed " +
                            "BACKGROUND_MAX_TIME_RECONNECT： $BACKGROUND_MAX_TIME_RECONNECT"
                )
                return
            }

            // 如果前面一个还存在，重连的时候先关闭前面一个
            close(true, "重连：先关闭连接，再重连")
            // 重连次数
            mCurReconnectCount.incrementAndGet()
            // 连接
            connect(tryLog)
        }
    }
}