package com.joyy.hagorpc.internal

import com.joyy.hagorpc.IFrequencyExecutor
import com.joyy.hagorpc.NotifyBundle
import com.joyy.hagorpc.birdge.LoggerBridge
import com.joyy.hagorpc.birdge.RunnableBridge
import com.joyy.hagorpc.birdge.SystemClockBridge
import kotlinx.atomicfu.locks.SynchronizedObject
import kotlinx.atomicfu.locks.synchronized
import kotlin.jvm.Volatile

class OneNotifyFrequencyHandler(
    serviceName: String, uri: Int, finalHandler: INotifyHandler,
    duration: Long,
) : INotifyHandler {
    private val mTag: String
    private val mService: String
    private val mUri: Int
    private val mFinalHandler: INotifyHandler

    private val lock = SynchronizedObject()

    // 等待通知的队列
    private val mWaitNotifyList: MutableList<NotifyBundle> = ArrayList()

    // 等待通知队列的复制
    private val mNotifyCopyList: MutableList<NotifyBundle> = ArrayList()

    // 实际触发通知的行为
    private var mActualNotifyAction: RunnableBridge? = null

    //数据收集
    @Volatile
    private var mLastNotifyTime: Long = 0

    @Volatile
    private var mReceiveTime: Long = 0

    //数据收集 end
    private val mFrequencyExecutor: IFrequencyExecutor
    private val mDuration: Long

    init {
        mTag = "NotifyFrequencyHandler_" + serviceName + "_" + uri
        mService = serviceName
        mUri = uri
        mFinalHandler = finalHandler
        mDuration = duration
        mFrequencyExecutor = RPCCore.mNotifyFrequencyDelegate.createExecutor(duration)
        LoggerBridge.logI(mTag, "init frequency duration: $duration")
    }

    override fun notify(bundle: NotifyBundle) {
        val frequencyDelegate = RPCCore.mNotifyFrequencyDelegate
        synchronized(lock) {

            //处理可替换和重置
            if (frequencyDelegate.canReplace(mService, mUri)) {
                for (item in mWaitNotifyList) {
                    if (item.protocol.uri == bundle.protocol.uri
                        && item.protocol.header.sname == bundle.protocol.header.sname
                    ) {
                        mWaitNotifyList.remove(item)
                        LoggerBridge.logI(
                            mTag,
                            "remove duplicate item: ${item.protocol.uri}, seqid: ${bundle.protocol.header.seqid}",
                        )
                        break
                    }
                }
            }
            mWaitNotifyList.add(bundle)
            LoggerBridge.logD(
                mTag, "cache size: ${mWaitNotifyList.size}, uri: ${bundle.protocol.uri}",
            )
            if (mReceiveTime == 0L) {
                mReceiveTime = SystemClockBridge.elapsedRealtime()
            }
        }
        var actualNotifyAction = mActualNotifyAction
        if (actualNotifyAction == null) {
            actualNotifyAction = RunnableBridge {
                synchronized(lock) {
                    mNotifyCopyList.clear()
                    mNotifyCopyList.addAll(mWaitNotifyList)
                    mWaitNotifyList.clear()
                }
                var notifyDataToHandler = frequencyDelegate.intercept(mNotifyCopyList)
                LoggerBridge.logD(
                    mTag,
                    "actual notify action run, notify the waited list: ${notifyDataToHandler.size}"
                )
                if (notifyDataToHandler.size > 1) {
                    val batchNum = frequencyDelegate.onBatch(mService, mUri)
                    if (batchNum < 0) {
                        for (b in notifyDataToHandler) {
                            mFinalHandler.notify(b)
                        }
                    } else {
                        if (notifyDataToHandler.size > batchNum) {
                            notifyDataToHandler = notifyDataToHandler.subList(
                                notifyDataToHandler.size - batchNum,
                                notifyDataToHandler.size
                            )
                        }
                        mFinalHandler.notify(notifyDataToHandler)
                    }
                } else {
                    for (b in notifyDataToHandler) {
                        mFinalHandler.notify(b)
                    }
                }
                val thisNotifyTime = SystemClockBridge.elapsedRealtime()
                frequencyDelegate.onAfterNotify(
                    bundle.protocol.header.sname,
                    obtainUris(mNotifyCopyList),
                    mLastNotifyTime,
                    thisNotifyTime,
                    mReceiveTime
                )
                mLastNotifyTime = thisNotifyTime
                mReceiveTime = 0
                mNotifyCopyList.clear()
            }
            mActualNotifyAction = actualNotifyAction
        }
        mFrequencyExecutor.execute(actualNotifyAction)
    }

    override fun notify(bundleList: List<NotifyBundle>) {
        // Do nothing
    }

    fun updateTime(time: Long) {
        if (mDuration != time) {
            mFrequencyExecutor.update(time)
            LoggerBridge.logI(mTag, "updateTime: $time")
        }
    }

    private fun obtainUris(notifyList: List<NotifyBundle>): List<Int> {
        val uris: MutableList<Int> = ArrayList(notifyList.size)
        for ((protocol) in notifyList) {
            uris.add(protocol.uri)
        }
        return uris
    }

    fun hasNotifyNotHandler(): Boolean {
        synchronized(lock) { return mWaitNotifyList.size > 0 }
    }
}