package com.joyy.hagorpc.internal

import com.joyy.hagorpc.IRPCDataProvider
import com.joyy.hagorpc.IRPCEventListener
import com.joyy.hagorpc.RPCCallException
import com.joyy.hagorpc.RPCCallRequest
import com.joyy.hagorpc.RPCCallResponse
import com.joyy.hagorpc.SafeRunnable
import com.joyy.hagorpc.birdge.ExecutorBridge
import com.joyy.hagorpc.birdge.LoggerBridge
import com.joyy.hagorpc.birdge.RPCPacketBridge
import com.joyy.hagorpc.birdge.RunnableBridge
import com.joyy.hagorpc.birdge.SystemBridge
import com.joyy.hagorpc.birdge.SystemClockBridge
import kotlin.jvm.Volatile

internal class HeartBeatTask(
    url: String?,
    from: Int?,
) : RunnableBridge {

    companion object {
        private const val TAG = "RPCCore.HeartBeatTask"

        // 记录全局的上次正在玩的游戏
        @Volatile
        private var sLastPlayingGame: String? = null
    }

    private var mHeartInterval: Long = 15000 // 下次的心跳间隔，单位毫秒

    // 接收pong的时间戳
    @Volatile
    private var mAcceptPongTs = SystemClockBridge.elapsedRealtime()

    @Volatile
    internal var mCliSeq = 0
    private val mUrl: String
    private var mFrom // 心跳触发来源，默认是内部触发，为null
            : Int?

    init {
        mCliSeq = RPCCore.identifySeq()
        mUrl = url ?: ""
        mFrom = from
    }

    @Volatile
    private var mStopped = false

    fun stop() {
        LoggerBridge.logI(TAG, "stop: seq = $mCliSeq")
        mStopped = true
    }

    private abstract class HeartBeatCallback(
        private val heartInterval: Long,
        private val callFrom: Int?,
        private val requestProto: String,
        private val heartbeatInfo: IRPCDataProvider.HeartbeatInfo,
        private val shouldSendHeart: () -> Boolean,
        private val sendOnlineProto: (callFrom: Int?, heartbeatInfo: IRPCDataProvider.HeartbeatInfo, fromRetry: Boolean) -> Unit
    ) : RPCCore.RPCCallback {

        override fun getTimeout(): Long {
            return heartbeatInfo.requestTimeout
        }

        override fun onRequest(req: RPCCallRequest) {
            super.onRequest(req)
            ExecutorBridge.getMainExecutor().execute {
                RPCCore.mEventListener.onStartHeartBeat(
                    req,
                    heartInterval,
                    requestProto,
                    callFrom
                )
            }
        }

        override fun onError(error: RPCCallException): Boolean {
            LoggerBridge.logE(
                TAG,
                "HeartBeat onError code: ${error.getCode()}, reason: ${error.getReason()}",
            )
            val toRetry =
                RPCCore.mEventListener.onHeartBeatError(error, false, callFrom)
            if (shouldSendHeart() && toRetry) {
                //失败后，立即重试一次
                sendOnlineProto(callFrom, heartbeatInfo, true)
            }
            return false
        }
    }

    override fun run() {

        // 每次心跳任务运行时获取当次的心跳信息
        val heartbeatInfo = RPCCore.mHeartbeatInfo.invoke(mFrom)
        LoggerBridge.logI(TAG, "HeartBeat task run heartInterval = $mHeartInterval")
        if (!shouldSendHeart()) {
            return
        }

        // 超过两次心跳间隔没有收到心跳回
        val currentTimeMillis = SystemClockBridge.elapsedRealtime()
        var pongTimeout = heartbeatInfo.pongTimeout
        if (pongTimeout <= 0) {
            pongTimeout = RPCConst.PONG_TIMEOUT_NUM * mHeartInterval + 3000
        }
        if (currentTimeMillis - mAcceptPongTs > pongTimeout) {
            // 发起重连吧
            LoggerBridge.logE(
                TAG,
                "cur: $currentTimeMillis tryReconnect because of no-pong since from $mAcceptPongTs (${currentTimeMillis - mAcceptPongTs}), pong: $mHeartInterval",
            )
            RPCCore.createReopenTask("发起重连-心跳超时", 0)
            return
        }

        // 后台配置用哪种协议发送心跳
        //按照服务器的要求，将心跳迁移到Online服务，这里加一个开关可以回滚，以防万一
        if (heartbeatInfo.useCProxy) {
            sendCproxyProto(mFrom, heartbeatInfo)
        } else {
            sendOnlineProto(mFrom, heartbeatInfo, false)
        }

        // 执行下一次心跳
        mFrom = null // 执行下一次心跳后来源要重置
        ExecutorBridge.getAsyncExecutor().executeDelay(
            this,
            mHeartInterval
        )
    }

    private fun shouldSendHeart(): Boolean {
        if (mStopped) {
            LoggerBridge.logE(TAG, "HeartBeat has stopped! seq = $mCliSeq")
            return false
        }

        // 已经被关闭了
        if (!RPCCore.checkConnected()) {
            LoggerBridge.logE(TAG, "HeartBeat client closed! seq = $mCliSeq")
            return false
        }
        if (mCliSeq != RPCCore.identifySeq()) {
            LoggerBridge.logE(TAG, "HeartBeat client seq not equal! seq = $mCliSeq")
            return false
        }
        return true
    }

    private fun sendCproxyProto(callFrom: Int?, heartbeatInfo: IRPCDataProvider.HeartbeatInfo) {
        val playingGame = heartbeatInfo.playingGame
        var playingGameChanged = false
        if (!sLastPlayingGame.isNullOrEmpty() && !sLastPlayingGame.equals(playingGame)
        ) {
            playingGameChanged = true
        }
        sLastPlayingGame = playingGame
        if (mStopped) {
            LoggerBridge.logE(TAG, "HeartBeat has stopped, no need to continue sending!")
            return
        }
        val header = RPCCore.getHeaderBuilder(RPCServiceDefine.CPROXY, false)
            .setBackGround(RPCCore.mIsBackground.value)
        val cproxy = RPCPacketBridge.fromInnerV2(
            RPCPacketBridge.createCProxyProto(
                playingGame,
                playingGameChanged,
                header
            )
        ) ?: kotlin.run {
            LoggerBridge.logE(TAG, "HeartBeat cproxy is null!")
            return
        }
        LoggerBridge.logI(
            TAG, "HeartBeat send heart beat task: ${cproxy.header.seqid}, from: $callFrom"
        )
        val start = SystemBridge.currentTimeMillis()
        // 发起心跳
        RPCCore.send(
            packet = cproxy,
            checkByHttp = false,
            callback = object :
                HeartBeatCallback(
                    mHeartInterval,
                    callFrom,
                    "CproxyProto",
                    heartbeatInfo,
                    shouldSendHeart = { shouldSendHeart() },
                    sendOnlineProto = { callFrom: Int?, heartbeatInfo: IRPCDataProvider.HeartbeatInfo, fromRetry: Boolean ->
                        sendOnlineProto(
                            callFrom,
                            heartbeatInfo,
                            fromRetry
                        )
                    }) {
                override fun onSuccess(res: RPCCallResponse) {
                    val handleStartTs = SystemBridge.currentTimeMillis()
                    val dispatchDuration = SystemBridge.currentTimeMillis() - res.dispatchTimestamp
                    val timeNow = SystemClockBridge.elapsedRealtime()
                    val heartBeatRes = RPCPacketBridge.heartBeatResFromCProxy(res.payload)
                    mHeartInterval = heartBeatRes.next * 1000
                    LoggerBridge.logI(
                        TAG,
                        "HeartBeat onResponse mHeartTime: $mHeartInterval, " +
                                "mServiceTime: ${heartBeatRes.timestamp}, mClientRealTime: $timeNow, duration: ${SystemBridge.currentTimeMillis() - start}, callFrom: $callFrom",
                    )
                    if (mHeartInterval <= 0) {
                        mHeartInterval = 10000
                    }
                    mAcceptPongTs = timeNow
                    val heartBeatData = IRPCEventListener.HeartBeatData(
                        mHeartInterval, heartBeatRes.timestamp, timeNow, dispatchDuration,
                        SystemBridge.currentTimeMillis() - handleStartTs, false
                    )
                    ExecutorBridge.getMainExecutor().execute {
                        RPCCore.mEventListener.onHeartBeatResponse(res, heartBeatData, callFrom)
                    }
                }
            }
        )

    }

    /**
     * 添加的心跳参数，需要使用 "|" 分割，每个位置对应值没有的时候，也要填 | + "", 不能没有
     * 例如 A|B|C|D, 没有B 的时候, 填 A||C|D
     */
    private fun sendOnlineProto(
        callFrom: Int?,
        heartbeatInfo: IRPCDataProvider.HeartbeatInfo,
        fromRetry: Boolean
    ) {
        val playingGame = heartbeatInfo.playingGame
        var playingGameChanged = false
        if (!sLastPlayingGame.isNullOrEmpty() && !sLastPlayingGame.equals(playingGame)
        ) {
            playingGameChanged = true
        }
        sLastPlayingGame = playingGame
        val bizMap = heartbeatInfo.onlineBizMap
        val header = RPCCore.getHeaderBuilder(RPCServiceDefine.IKXD_ONLINE, false)
            .setBackGround(RPCCore.mIsBackground.value)
        val payload = RPCPacketBridge.createOnlineProto(
            playingGame,
            playingGameChanged,
            header,
            bizMap
        )
        val onlineProto = RPCPacketBridge.fromInnerV2(
            payload
        ) ?: kotlin.run {
            LoggerBridge.logE(TAG, "HeartBeat has stopped, no need to continue sending!")
            return
        }
        onlineProto.toByteArray = { payload }
        if (mStopped) {
            LoggerBridge.logE(TAG, "HeartBeat has stopped!")
            return
        }
        LoggerBridge.logI(
            TAG, "HeartBeat send heart beat task: ${onlineProto.header.seqid}, from: $callFrom",
        )
        RPCCore.send(
            packet = onlineProto,
            checkByHttp = false,
            callback = object :
                HeartBeatCallback(
                    mHeartInterval,
                    callFrom,
                    "OnlineProto",
                    heartbeatInfo,
                    shouldSendHeart = { shouldSendHeart() },
                    sendOnlineProto = { callFrom: Int?, heartbeatInfo: IRPCDataProvider.HeartbeatInfo, fromRetry: Boolean ->
                        sendOnlineProto(
                            callFrom,
                            heartbeatInfo,
                            fromRetry
                        )
                    }) {
                override fun onSuccess(res: RPCCallResponse) {
                    val handleStartTs = SystemBridge.currentTimeMillis()
                    val dispatchDuration = SystemBridge.currentTimeMillis() - res.dispatchTimestamp
                    val timeNow = SystemClockBridge.elapsedRealtime()
                    val heartBeatRes = RPCPacketBridge.heartBeatResFromOnline(res.payload)
                    mHeartInterval = heartBeatRes.next * 1000
                    LoggerBridge.logI(
                        TAG,
                        "HeartBeat onResponse mHeartTime: $mHeartInterval, " +
                                "mServiceTime: ${heartBeatRes.timestamp}, mClientRealTime: $timeNow, callFrom: $callFrom",
                    )
                    if (mHeartInterval <= 0) {
                        mHeartInterval = 10000
                    }
                    mAcceptPongTs = timeNow
                    val heartBeatData = IRPCEventListener.HeartBeatData(
                        mHeartInterval, heartBeatRes.timestamp, timeNow, dispatchDuration,
                        SystemBridge.currentTimeMillis() - handleStartTs, false
                    )
                    ExecutorBridge.getMainExecutor().execute {
                        RPCCore.mEventListener.onHeartBeatResponse(res, heartBeatData, callFrom)
                    }
                }
            }
        )

    }
}