package com.joyy.hagorpc.internal

import com.joyy.hagorpc.IRPCDispatcher
import com.joyy.hagorpc.IRPCNotify
import com.joyy.hagorpc.IRegisterId
import com.joyy.hagorpc.NotifyBundle
import com.joyy.hagorpc.SafeRunnable
import com.joyy.hagorpc.birdge.ExecutorBridge
import com.joyy.hagorpc.birdge.LoggerBridge
import kotlinx.atomicfu.atomic
import kotlinx.atomicfu.locks.SynchronizedObject
import kotlinx.atomicfu.locks.synchronized

/**
 * 广播分发器基类
 * Created by wjh on 2021/10/22
 */
internal abstract class AbsRPCDispatcher(
) : IRPCDispatcher {

    private val lock = SynchronizedObject()

    private val mObserver = mutableMapOf<IRegisterId, MutableList<IRPCNotify>>()

    abstract fun logTag(): String

    abstract fun findAction(
        bundle: NotifyBundle,
        observerMap: Map<IRegisterId, List<IRPCNotify>>
    ): List<NotifyAction>

    override fun register(id: IRegisterId, notify: IRPCNotify) {
        synchronized(lock) {
            var list = mObserver[id]
            if (list == null) {
                list = mutableListOf()
                mObserver[id] = list
            }
            if (!list.contains(notify)) {
                list.add(notify)
            }
        }
    }

    override fun unregister(id: IRegisterId, notify: IRPCNotify) {
        synchronized(lock) {
            val list = mObserver[id]
            list?.remove(notify)
        }
    }

    override fun notify(bundle: NotifyBundle) {
        val actionList: List<NotifyAction> = synchronized(lock) {
            findAction(bundle, mObserver)
        }
        actionList.forEach {
            innerNotify(it)
        }
    }

    override fun notify(bundleList: List<NotifyBundle>) {
        if (bundleList.isEmpty()) {
            LoggerBridge.logD(logTag(), "notify ignore, bundle list is empty")
            return
        }

        val actionList = mutableListOf<NotifyAction>()
        bundleList.forEach { bundle ->
            synchronized(lock) {
                actionList.addAll(findAction(bundle, mObserver))
            }
        }

        actionList.forEach {
            innerNotify(it)
        }
    }

    private fun innerNotify(action: NotifyAction) {

        val target = when {
            action.uiObserver.isNotEmpty() && action.threadObserver.isNotEmpty() -> 2
            action.uiObserver.isNotEmpty() || action.threadObserver.isNotEmpty() -> 1
            else -> 0
        }
        val count = atomic(0)

        // 通知完要回收
        fun recycle(action: NotifyAction) {
            LoggerBridge.logD(logTag(), "notifyAction try recycle")
            if (count.incrementAndGet() == target) {
                LoggerBridge.logD(logTag(), "notifyAction recycle finally")
                action.recycle()
            }
        }

        if (target == 0) {
            recycle(action)
            return
        }

        LoggerBridge.logD(
            logTag(),
            "inner notify action: $action, wait recycle flag: $target"
        )

        if (action.uiObserver.isNotEmpty()) {
            ExecutorBridge.getMainExecutor().execute(
                SafeRunnable(
                    task = {
                        action.uiObserver.forEach {
                            it.onNotify(
                                action.payload,
                                action.extra
                            )
                        }
                        recycle(action)
                    })
            )
        }
        if (action.threadObserver.isNotEmpty()) {
            ExecutorBridge.getAsyncExecutor().execute(
                SafeRunnable(
                    task = {
                        action.threadObserver.forEach {
                            it.onNotify(
                                action.payload,
                                action.extra
                            )
                        }
                        recycle(action)
                    })
            )
        }
    }
}