package com.joyy.hagorpc.birdge

import com.joyy.hagorpc.INetworkClient
import com.joyy.hagorpc.RPCManager
import java.util.concurrent.TimeUnit

actual object WsClientBridge {
    actual fun newWebSocket(
        wsUrl: String,
        header: Map<String, String>,
        listener: WebSocketListener
    ): WebSocket {
        return AndroidWebSocket(wsUrl, header, listener)
    }

    actual interface WebSocketListener {
        actual fun onOpen()
        actual fun onMessage(bytes: ByteArray)
        actual fun onClose(code: Int, reason: String)
        actual fun onError(errorMessage: String)

    }

    actual interface WebSocket {
        actual fun getWsUrl(): String
        actual fun close(code: Int, reason: String)
        actual fun send(data: ByteArray): Boolean
        actual fun queueSize(): Long

    }

    class AndroidWebSocket(
        private val wsUrl: String,
        private val header: Map<String, String>,
        listener: WebSocketListener
    ) : WebSocket {
        private val websocket: INetworkClient.IWebSocket

        init {
            val connectTimeout = RPCManager.mInternalContext.getConnectTimeout()
            val pingInterval = RPCManager.mInternalContext.getPingInterval()
            val client = RPCManager.mInternalContext.getNetworkClient().newBuilder()
                .connectTimeout(connectTimeout, TimeUnit.MILLISECONDS)
                .pingInterval(pingInterval)
                .connectionPool(1, 3600, TimeUnit.SECONDS)
                .build()
            val request = INetworkClient.Request.Builder()
                .get()
                .addHeader(header)
                .url(wsUrl)
                .build()
            websocket = client.newWebSocket(request, object : INetworkClient.IWebSocketListener {
                override fun onOpen() {
                    listener.onOpen()
                }

                override fun onMessage(bytes: ByteArray?) {
                    listener.onMessage(bytes ?: ByteArray(0))
                }

                override fun onClosed(
                    code: Int, reason: String?
                ) {
                    listener.onClose(code, reason ?: "")
                }

                override fun onFailure(t: Throwable?) {
                    listener.onError(t?.message ?: "")
                }
            })
        }

        override fun getWsUrl(): String {
            return wsUrl
        }

        override fun close(code: Int, reason: String) {
            websocket.close(code, reason)
        }

        override fun send(data: ByteArray): Boolean {
            return websocket.send(data)
        }

        override fun queueSize(): Long {
            return websocket.queueSize()
        }

    }

}