package com.joyy.hagorpc

import kotlin.jvm.JvmField

/**
 * 下行包，针对可靠消息cproxy会额外封装一层自己的header。
 * 客户端解包时，先用InnerV3解析下行二进制数据，如果解析成功，则根据version版本来解析payload
 * 如果用InnerV3解析失败，需要InnerV2解析
 * @param seqId 当msgtype是可靠消息类型时，客户端需要保存seqid。用于重连后拉取
 * @param msgType 枚举值，见上述EMsgType。
 * @param version 协议解包方式版本。见EMsgUnpackType
 * @param magic 16进制：为避免与旧的magic冲突，使用0x4861676e(0x4861676f - 1)
 */
data class RPCPacketV3(
    @JvmField val seqId: Long = 0,
    @JvmField val msgType: Int = 0,
    @JvmField val version: Int = 0,
    @JvmField val magic: Long = 0,
    @JvmField val payload: ByteArray = ByteArray(0)
) {


    enum class EMsgUnpackType(val value: Int) {
        MSG_UNPACK_NONE(0),
        MSG_UNPACK_INNERV2(1), // 用innerv2来解析payload
        UNRECOGNIZED(-1)
    }

    enum class EMsgType(val value: Int) {
        MSG_NONE(0),
        MSG_DOWNSTREAM_RELIABLE(1), // 下行可靠消息
        UNRECOGNIZED(-1)
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other == null || this::class != other::class) return false

        other as RPCPacketV3

        if (seqId != other.seqId) return false
        if (msgType != other.msgType) return false
        if (version != other.version) return false
        if (magic != other.magic) return false
        if (!payload.contentEquals(other.payload)) return false

        return true
    }

    override fun hashCode(): Int {
        var result = seqId.hashCode()
        result = 31 * result + msgType
        result = 31 * result + version
        result = 31 * result + magic.hashCode()
        result = 31 * result + payload.contentHashCode()
        return result
    }

    override fun toString(): String {
        return "RPCPacketV3(seqId=$seqId, msgType=$msgType, version=$version, magic=$magic)"
    }
}
