package com.joyy.hagorpc

import android.util.Log
import com.joyy.hagorpc.birdge.LocalPersistenceBridge
import com.joyy.hagorpc.impl.DefaultConfigHelper
import com.joyy.hagorpc.impl.DefaultNetworkDelegate
import com.joyy.hagorpc.internal.*
import com.joyy.hagorpc.internal.RPCPacketExt.fromInnerV2
import okio.ByteString.Companion.encodeUtf8
import java.util.concurrent.atomic.AtomicBoolean

/**
 * rpc处理管理类
 * Created by wjh on 2021/10/22
 */
@Suppress("unused")
object RPCManager : IRPCSender, IRPCReceiver {

    private const val TAG = "RPCManager"

    private lateinit var mProvider: IRPCDataProvider
    private lateinit var mConfig: RPCConfig

    // 当前状态
    private var mStatus: WsStatus? = null

    // 长连接使用的网络库
    private var mWsNetwork: String? = null

    // 是否触发建立长连接
    private var mCallConnect: Boolean = false

    private val mStateListenerList = mutableListOf<IRPCStateListener>()

    private val mInit = AtomicBoolean(false)

    /**
     * 内部上下文
     */
    internal val mInternalContext: IRPCInternalContext by lazy {
        object : IRPCInternalContext {
            override fun getToken(): String {
                return this@RPCManager.getToken() ?: ""
            }

            override fun getUid(): String? {
                return this@RPCManager.getDataProvider().uid()
            }

            override fun getNetworkClient(): INetworkClient {
                return mConfig.mNetworkClient
            }

            override fun getPingInterval(): Long {
                return mConfig.mPingInterval
            }

            override fun getConnectTimeout(): Long {
                return mConfig.mConnectTimeout
            }

            override fun getConnectRetryMax(): Int {
                return mConfig.mConnectRetryMax
            }

            override fun getRequestMaxRetry(): Int {
                return mConfig.mRequestRetryMax
            }

            override fun getCommonHeader(wsUrl: String, forWs: Boolean): Map<String, String>? {
                return this@RPCManager.getDataProvider().commonHeader(wsUrl, forWs)
            }

            override fun useGZip(wsUrl: String): Boolean {
                return mConfig.mUseGZip
            }

            override fun useReliableBroadcast(): Boolean {
                return mConfig.mReliableBroadcast
            }

            override fun enableBackgroundReconnect(wsUrl: String): Boolean {
                return mProvider.enableBackgroundReconnect(wsUrl)
            }

            override fun getLanguage(): String? {
                return mProvider.language()
            }

            override fun getCountry(): String? {
                return mProvider.country()
            }

            override fun getHeartbeatInfo(from: Int?): IRPCDataProvider.HeartbeatInfo {
                return mProvider.heartbeatInfo(from)
            }

            override fun getRouteKey(service: String): String {
                return getDataProvider().routeKey(service) ?: ""
            }

            override fun getMainExecutor(): IRPCExecutor {
                return this@RPCManager.getMainExecutor()
            }

            override fun getAsyncExecutor(): IRPCExecutor {
                return this@RPCManager.getAsyncExecutor()
            }

            override fun getNetworkDelegate(): IRPCNetworkDelegate {
                return this@RPCManager.getNetworkDelegate()
            }

            override fun getMemoryDelegate(): IRPCMemoryDelegate {
                return mConfig.mMemoryDelegate
            }

            override fun getEventListener(): IRPCEventListener {
                return mConfig.mEventListener
            }

            override fun getNotifyFrequencyDelegate(): INotifyFrequencyDelegate {
                return mConfig.mNotifyFrequencyDelegate
            }

            override fun getLogger(): ILogger {
                return this@RPCManager.getLogger()
            }

            override fun sendProtoBytes(payload: ByteArray, callback: IResCallback) {
                return this@RPCManager.send(payload, callback)
            }

            override fun updateWsClientNetwork(url: String, network: String?) {
                synchronized(this@RPCManager) {
                    mWsNetwork = network
                }
            }
        }
    }

    private val mWsClientCallback: RPCCore.WsCallback by lazy {
        object : RPCCore.WsCallback {
            override fun onConnectSucceed(wsUrl: String) {
                updateStatus(RPCCore.status, null, null)
            }

            override fun onReConnectMax(wsUrl: String) {
                // Do nothing
            }

            override fun onConnecting(wsUrl: String) {
                updateStatus(RPCCore.status, null, null)
            }

            override fun onResponse(wsUrl: String, bytes: ByteArray) {
                // Do nothing
            }

            override fun onError(wsUrl: String, code: Int, reason: String) {
                updateStatus(RPCCore.status, code, reason)
                if (getLogger().isDebuggable() && code >= WsCode.KICKOFF_CODE &&
                    code < WsCode.TOKEN_PARSE_FAIL
                ) {
                    throw RuntimeException("WsClient onError, code:$code")
                }
            }

            override fun onDisconnect(
                wsUrl: String,
                code: Int,
                reason: String
            ) {
                updateStatus(RPCCore.status, code, reason)
            }
        }
    }

    /**
     * 初始化
     */
    @Synchronized
    fun init(dataProvider: IRPCDataProvider, config: RPCConfig) {
        if (!mInit.get()) {
            mProvider = dataProvider
            mConfig = RPCConfig(config)
            mInit.set(true)
            mConfig.getContext()?.let {
                val networkDelegate = mConfig.mNetworkDelegate
                if (networkDelegate is DefaultNetworkDelegate) {
                    networkDelegate.start(it)
                }
                LocalPersistenceBridge.init(it)
            }

            RPCCore.config(
                token = { mInternalContext.getToken() },
                wsUrl = { getWsUrl() ?: "" },
                httpUrl = { getHttpUrl() ?: "" },
                uid = { mInternalContext.getUid() ?: "" },
                enableBackgroundReconnect = { wsUrl ->
                    mInternalContext.enableBackgroundReconnect(
                        wsUrl
                    )
                },
                useGZip = { wsUrl -> mInternalContext.useGZip(wsUrl) },
                useReliableBroadcast = { _ -> mInternalContext.useReliableBroadcast() },
                requestMaxRetry = mInternalContext.getRequestMaxRetry(),
                connectRetryMax = mInternalContext.getConnectRetryMax(),
                connectTimeout = mInternalContext.getConnectTimeout(),
                pingInterval = mInternalContext.getPingInterval(),
                eventListener = mInternalContext.getEventListener(),
                resendIfConnected = mConfig.mResendIfConnected,
                heartbeatInfo = { from -> mInternalContext.getHeartbeatInfo(from) },
                memoryDelegate = mInternalContext.getMemoryDelegate(),
                notifyFrequencyDelegate = mInternalContext.getNotifyFrequencyDelegate(),
                dispatchRequestToHttp = { condition ->
                    getDataProvider().dispatchRequestToHttp(
                        condition
                    )
                },
                commonHeader = { wsUrl, forWs ->
                    mInternalContext.getCommonHeader(wsUrl, forWs) ?: emptyMap()
                },
            )

            getLogger().logI(TAG, "init version[${BuildConfig.SdkVersion}]")
            getLogger().logD(TAG, "init config: $config")
        }
    }

    @Synchronized
    private fun checkInit() {
        if (!mInit.get()) {
            throw AssertionError("not init")
        }
    }

    @Synchronized
    fun connect(force: Boolean = false) {
        checkInit()
        mCallConnect = true

        RPCCore.connectWebSocket(
            forceReConnect = force,
            callback = mWsClientCallback
        )
    }

    @Synchronized
    fun disconnect() {
        checkInit()
        mCallConnect = false
        RPCCore.disconnectWebSocket()
    }

    @Synchronized
    fun getStatus(): WsStatus {
        return RPCCore.status
    }

    @Synchronized
    fun getWsNetwork(): String? {
        return mWsNetwork
    }

    @Synchronized
    fun getWsUrl(): String? {
        checkInit()
        return getDataProvider().url()
    }

    @Synchronized
    fun getHttpUrl(): String? {
        checkInit()
        val wsUrl = getDataProvider().url()
        if (!wsUrl.isNullOrEmpty()) {
            return if (wsUrl.regionMatches(0, "ws:", 0, 3, ignoreCase = true)) {
                "http:" + wsUrl.substring(3) + RPCConst.RPC_HTTP_API
            } else if (wsUrl.regionMatches(0, "wss:", 0, 4, ignoreCase = true)) {
                "https:" + wsUrl.substring(4) + RPCConst.RPC_HTTP_API
            } else {
                wsUrl + RPCConst.RPC_HTTP_API
            }
        }
        return wsUrl
    }

    @Synchronized
    fun getToken(): String? {
        checkInit()
        return getDataProvider().token()
    }

    fun isNetAvailable(): Boolean {
        checkInit()
        return getNetworkDelegate().isNetworkAvailable()
    }

    private fun getRegisterUri(uri: Any?): Long? {
        return when (uri) {
            is Long -> {
                uri
            }
            is Int -> {
                uri.toLong()
            }
            else -> {
                null
            }
        }
    }

    override fun register(sname: String, uri: Any?, notify: IRPCNotify) {
        checkInit()
        RPCCore.register(
            RegisterId(
                sname, getRegisterUri(uri), true
            ), notify
        )
    }

    override fun unregister(sname: String, uri: Any?, notify: IRPCNotify) {
        checkInit()
        RPCCore.unregister(
            RegisterId(
                sname, getRegisterUri(uri), true
            ), notify
        )
    }

    override fun registerForFuzzy(sname: String, uri: Any?, notify: IRPCNotify) {
        checkInit()
        RPCCore.register(
            RegisterId(
                sname, getRegisterUri(uri), false
            ), notify
        )
    }

    override fun unregisterForFuzzy(sname: String, uri: Any?, notify: IRPCNotify) {
        checkInit()
        RPCCore.unregister(
            RegisterId(
                sname, getRegisterUri(uri), false
            ), notify
        )
    }

    override fun listenStatus(listener: IRPCStateListener) {
        checkInit()
        getLogger().logD(TAG, "listenStatus")
        synchronized(mStateListenerList) {
            if (!mStateListenerList.contains(listener)) {
                mStateListenerList.add(listener)
            }
            if (getLogger().isDebuggable()) {
                Log.d(TAG, "stateListenerList size: ${mStateListenerList.size}")
            }
        }
    }

    override fun removeListen(listener: IRPCStateListener) {
        checkInit()
        getLogger().logD(TAG, "removeListenStatus")
        synchronized(mStateListenerList) {
            mStateListenerList.remove(listener)

            if (getLogger().isDebuggable()) {
                Log.d(TAG, "stateListenerList size: ${mStateListenerList.size}")
            }
        }
    }

    override fun send(data: ByteArray, callback: IResCallback?) {
        checkInit()
        val task = Runnable {
            val innerV2 = RPCProtoHelper.parseInner(data)
            if (innerV2?.header == null) {
                val msg = "parse inner v2 error"
                getLogger().logE(TAG, msg)
                callback?.onError(
                    RPCCallException(
                        null,
                        canRetry = false,
                        isTimeout = false,
                        reason = msg,
                        code = WsCode.UNKNOWN_CODE,
                        responseHeader = null
                    )
                )
            } else {
                innerSend(innerV2.fromInnerV2(), callback)
            }
        }
        getAsyncExecutor().execute(task)
    }

    override fun send(
        sname: String, method: String?, data: ByteArray,
        callback: IResCallback?
    ) {
        send(sname, method, null, data, callback)
    }

    override fun send(
        sname: String, method: String?, roomId: String?, data: ByteArray,
        callback: IResCallback?
    ) {
        checkInit()
        val task = Runnable {
            innerSend(sname, method, roomId, data, callback)
        }
        getAsyncExecutor().execute(task)
    }

    override fun triggerHeartbeat(from: Int?) {
        checkInit()
        getLogger().logI(TAG, "trigger heartbeat from: $from")
        RPCCore.triggerHeartbeat(from)
    }

    /**
     * 更新对应service的路由规则
     */
    fun updateRoutingKey(service: String?, key: String?) {
        if (service.isNullOrEmpty()) {
            return
        }
        RPCLocalPersistence.updateCProxyRouting(service, key ?: "")
    }

    /**
     * 获取对应service的路由规则
     */
    fun getRoutingKey(service: String?): String {
        if (service.isNullOrEmpty()) {
            return ""
        }
        return RPCLocalPersistence.getCProxyRouting(service)
    }

    /**
     * 切换到前台
     */
    fun changeToForeground() {
        checkInit()
        getLogger().logI(TAG, "changeToForeground")
        RPCCore.changeToForeground()
    }

    /**
     * 切换到后台
     */
    fun changeToBackground() {
        checkInit()
        getLogger().logI(TAG, "changeToBackground")
        RPCCore.changeToBackground()
    }

    internal fun lang(): String {
        return DefaultConfigHelper.getLanguageAndCountry(
            getDataProvider().language(), getDataProvider().country()
        )
    }

    private fun innerSend(
        sname: String, method: String?, roomId: String?,
        data: ByteArray, callback: IResCallback?
    ) {
        val headerBuilder = RPCCore.getHeaderBuilder(sname, true)
            .setLang(lang())
            .setMethod(method ?: "")
            .setRoomid(roomId ?: "")
        val pcid = getDataProvider().pcid()
        if (!pcid.isNullOrEmpty()) {
            val extend: MutableMap<String, ByteArray> = HashMap(1)
            extend[RPCConst.X_PCID] = pcid.encodeUtf8().toByteArray()
            headerBuilder.extend(extend)
        }
        val header = headerBuilder.build()
        val innerV2 = RPCPacketV2(header, 0, data)
        innerSend(innerV2, callback)
    }

    private fun innerSend(
        protocol: RPCPacketV2,
        callback: IResCallback?,
    ) {
        WsRequest.obtain().apply {
            this.mContext = mInternalContext
            this.wsUrl = getWsUrl()
            this.httpUrl = getHttpUrl()
            this.callback = callback
            this.protocol = protocol

            innerSend(this)
        }
    }

    private fun getDataProvider(): IRPCDataProvider {
        return mProvider
    }

    private fun getMainExecutor(): IRPCExecutor {
        return mConfig.mMainExecutor
    }

    /**
     * 异步队列（每次只执行一个任务）
     */
    private fun getAsyncExecutor(): IRPCExecutor {
        return mConfig.mAsyncExecutor
    }

    private fun getNetworkDelegate(): IRPCNetworkDelegate {
        return mConfig.mNetworkDelegate
    }

    private fun getEventListener(): IRPCEventListener {
        return mConfig.mEventListener
    }

    private fun getLogger(): ILogger {
        return mConfig.mLogger
    }

    @Synchronized
    private fun updateStatus(status: WsStatus, reasonCode: Int?, reasonInfo: String?) {
        getLogger().logI(
            TAG,
            "update state: $status, reasonCode: $reasonCode, reasonInfo: $reasonInfo"
        )
        if (mStatus == status) {
            // 重复变更不通知上层
            return
        }
        mStatus = status
        val listenerList = synchronized(mStateListenerList) {
            mStateListenerList.toList()
        }

        getMainExecutor().execute {
            listenerList.forEach {
                it.onChange(status, reasonCode, reasonInfo)
            }
        }
    }

    /**
     * 实际发送请求行为
     */
    private fun innerSend(request: WsRequest) {
        RPCCore.send(
            packet = request.protocol ?: RPCPacketV2(
                RPCHeader.newBuilder().build(),
                0,
                ByteArray(0)
            ),
            callback = object : RPCCore.RPCCallback {
                override fun onRequest(req: RPCCallRequest) {
                    request.callback?.onRequest(req)
                }

                override fun onSuccess(response: RPCCallResponse) {
                    request.callback(response)
                }

                override fun onError(
                    error: RPCCallException
                ): Boolean {
                    return request.callback(error)
                }

                override fun expectHttpRequest(): Boolean {
                    return request.callback?.expectHttpRequest() ?: false
                }

                override fun needToken(): Boolean {
                    return request.callback?.needToken() ?: true
                }

                override fun getTimeout(): Long {
                    return request.callback?.getTimeout() ?: 0
                }
            }
        )
    }
}