package com.joyy.hagorpc

import kotlin.jvm.JvmField

/**
 * @param sname service name
 * @param seqid sequence id
 * @param code return code
 * @param lang 语言代码_国家代码，如:zh_CN，en_US
 * @param tid traceid 客户端不用填，只需要收到包的时候log出来
 * @param method rpc调用时使用的函数名称
 * @param version 版本信息, 客户端现在用这个字段来区分新老消息，老版本消息禁止填写
 */

class RPCHeader(
    @JvmField val sname: String,
    @JvmField val seqid: Long,
    @JvmField val code: Long,
    @JvmField val lang: String,
    @JvmField val tid: String?,
    @JvmField val roomid: String,
    @JvmField val method: String,
    @JvmField val version: String,
    @JvmField val msgtype: MSGTYPE,
    @JvmField val backGround: Boolean,
    @JvmField val biz: ByteArray?,
    @JvmField val extend: Map<String, ByteArray>,
    @JvmField val routingKey: ByteArray?,
    @JvmField val gameid: String?
) {
    class Builder {

        var sname: String = ""
        var seqid: Long = 0
        var code: Long = 0
        var lang: String = ""
        var tid: String? = null
        var roomid: String = ""
        var method: String = ""
        var version: String = ""
        var msgtype: MSGTYPE = MSGTYPE.MSGTYPE_UNKNOWN
        var backGround: Boolean = false
        var biz: ByteArray? = null
        var extend: MutableMap<String, ByteArray> = mutableMapOf()
        var routingKey: ByteArray? = null
        var gameid: String? = null


        fun setCode(value: Long): Builder = apply {
            this.code = value
        }

        fun setSeqid(value: Long): Builder = apply {
            this.seqid = value
        }

        fun setSname(value: String): Builder = apply {
            this.sname = value
        }

        fun setRoutingKey(value: ByteArray): Builder = apply {
            this.routingKey = value
        }

        fun setMsgType(value: MSGTYPE): Builder = apply {
            this.msgtype = value
        }

        fun setVersion(value: String): Builder = apply {
            this.version = value
        }

        fun setLang(value: String): Builder = apply {
            this.lang = value
        }

        fun setBackGround(value: Boolean): Builder = apply {
            this.backGround = value
        }

        fun setMethod(value: String): Builder = apply {
            this.method = value
        }

        fun setRoomid(value: String): Builder = apply {
            this.roomid = value
        }

        fun extend(value: Map<String, ByteArray>): Builder = apply {
            this.extend.putAll(value)
        }

        fun build(): RPCHeader {
            return RPCHeader(
                sname = sname,
                seqid = seqid,
                code = code,
                lang = lang,
                tid = tid,
                roomid = roomid,
                method = method,
                version = version,
                msgtype = msgtype,
                backGround = backGround,
                biz = biz,
                extend = extend,
                routingKey = routingKey,
                gameid = gameid
            )
        }
    }

    companion object {
        val EMPTY = RPCHeader(
            sname = "",
            seqid = 0,
            code = 0,
            lang = "",
            tid = "",
            roomid = "",
            method = "",
            version = "",
            msgtype = MSGTYPE.MSGTYPE_UNKNOWN,
            backGround = false,
            biz = null,
            extend = emptyMap(),
            routingKey = null,
            gameid = null
        )

        internal fun newBuilder(): Builder {
            return Builder()
        }

    }

    override fun toString(): String {
        return "RPCHeader(sname='$sname', seqid=$seqid, code=$code, lang='$lang', tid='$tid', roomid='$roomid', method='$method', version='$version', msgtype=$msgtype, backGround=$backGround, routingKey=${routingKey.contentToString()}, gameid='$gameid')"
    }

}


enum class MSGTYPE(val value: Int) {
    MSGTYPE_UNKNOWN(0),
    MSGTYPE_REQ(1), // 请求消息
    MSGTYPE_RESP(2), // 响应消息
    MSGTYPE_NOTICE(3), // 单播消息
    UNRECOGNIZED(-1) //
}

fun Int.fromValue(): MSGTYPE {
    return when (this) {
        0 -> MSGTYPE.MSGTYPE_UNKNOWN
        1 -> MSGTYPE.MSGTYPE_REQ
        2 -> MSGTYPE.MSGTYPE_RESP
        3 -> MSGTYPE.MSGTYPE_NOTICE
        else -> MSGTYPE.UNRECOGNIZED
    }
}
