package com.joyy.hagorpc

import android.content.Context
import com.joyy.hagorpc.impl.DefaultAsyncExecutor
import com.joyy.hagorpc.impl.DefaultConfigHelper
import com.joyy.hagorpc.impl.DefaultEventListener
import com.joyy.hagorpc.impl.DefaultLogger
import com.joyy.hagorpc.impl.DefaultMainExecutor
import com.joyy.hagorpc.impl.DefaultMemoryDelegate
import com.joyy.hagorpc.impl.DefaultNetworkDelegate
import com.joyy.hagorpc.impl.DefaultNotifyFrequency
import com.joyy.hagorpc.internal.GzipUtils
import com.joyy.hagorpc.internal.RPCConst
import com.joyy.hagorpc.internal.RPCProtoHelper
import java.lang.ref.Reference
import java.lang.ref.WeakReference

/**
 * RPC 通用配置（业务方可根据自身条件自定义）
 * Created by wjh on 2022/3/25
 */
class RPCConfig {
    private val mContextRef: Reference<Context>
    var mNetworkClient: INetworkClient
    var mReliableBroadcast = RPCConst.RELIABLE_BROADCAST
    var mPingInterval: Long = RPCConst.PING_INTERVAL
    var mConnectTimeout: Long = RPCConst.CONNECT_TIMEOUT
    var mConnectRetryMax: Int = RPCConst.CONNECT_RETRY_MAX
    var mRequestRetryMax = RPCConst.REQUEST_RETRY_MAX
    var mResendIfConnected = RPCConst.RESEND_IF_CONNECTED

    var mUseGZip = false
    // TODO: Android也需要增加业务层传入的配置信息，以做新通道处理。
    var mAppId: String = ""
    var mAppToken: () -> String = { "" }
    var mAppUid: String = ""

    var mAsyncExecutor: IRPCExecutor
    var mMainExecutor: IRPCExecutor

    var mNetworkDelegate: IRPCNetworkDelegate
    var mMemoryDelegate: IRPCMemoryDelegate

    var mNotifyFrequencyDelegate: INotifyFrequencyDelegate

    var mEventListener: IRPCEventListener

    var mLogger: ILogger
        set(value) {
            field = value
            GzipUtils.setLogger(field)
            RPCProtoHelper.setLogger(field)
        }

    constructor(context: Context, networkClient: INetworkClient) {
        mContextRef = WeakReference(context.applicationContext)
        mLogger = DefaultLogger()
        mNetworkClient = networkClient
        mMainExecutor = DefaultMainExecutor()
        mAsyncExecutor = DefaultAsyncExecutor()
        mNetworkDelegate = DefaultNetworkDelegate()
        mMemoryDelegate = DefaultMemoryDelegate()
        mNotifyFrequencyDelegate = DefaultNotifyFrequency()
        mEventListener = DefaultEventListener()
    }

    internal constructor(config: RPCConfig) {
        mContextRef = config.mContextRef
        mNetworkClient = config.mNetworkClient
        mReliableBroadcast = config.mReliableBroadcast
        mPingInterval = config.mPingInterval
        mConnectTimeout = config.mConnectTimeout
        mConnectRetryMax = config.mConnectRetryMax
        mRequestRetryMax = config.mRequestRetryMax
        mResendIfConnected = config.mResendIfConnected
        mUseGZip = config.mUseGZip
        mMainExecutor = config.mMainExecutor
        mAsyncExecutor = config.mAsyncExecutor
        mNetworkDelegate = config.mNetworkDelegate
        mMemoryDelegate = config.mMemoryDelegate
        mNotifyFrequencyDelegate = config.mNotifyFrequencyDelegate
        mEventListener = config.mEventListener
        mLogger = config.mLogger
    }

    fun getContext(): Context? {
        return mContextRef.get()
    }

    override fun toString(): String {
        return "RPCConfig(" +
                "mReliableBroadcast=$mReliableBroadcast, " +
                "mPingInterval=$mPingInterval, " +
                "mConnectTimeout=$mConnectTimeout, mConnectRetryMax=$mConnectRetryMax, " +
                "mRequestRetryMax=$mRequestRetryMax, mResendIfConnected=$mResendIfConnected, " +
                "mUseGZip=$mUseGZip, " +
                "mAsyncExecutor=${mAsyncExecutor.javaClass.name}, mMainExecutor=${mMainExecutor.javaClass.name}, " +
                "mNetworkDelegate=${mNetworkDelegate.javaClass.name}, " +
                "mMemoryDelegate=${mMemoryDelegate.javaClass.name}, " +
                "mNotifyFrequencyDelegate=${mNotifyFrequencyDelegate.javaClass.name}, " +
                "mEventListener=${mEventListener.javaClass.name}," +
                " mLogger=${mLogger.javaClass.name})"
    }
}