package com.joyy.hagorpc

import android.content.Context
import java.util.concurrent.TimeUnit

interface INetworkClient {

    fun newBuilder(): IBuilder

    fun newWebSocket(request: Request, listener: IWebSocketListener): IWebSocket

    fun newCall(request: Request): ICall

    fun getErrorCodeByExc(context: Context?, e: Throwable?): Int

    enum class HttpMethod {
        POST, GET
    }

    interface ICallback {
        fun onResponse(
            response: IResponse
        )

        fun onFailure(throwable: Throwable?)
    }

    interface ICall {
        fun enqueue(callback: ICallback)
    }

    interface IResponse {
        fun code(): Int
        fun headersToString(): String
        fun isSuccessful(): Boolean
        fun responseHeaderFields(): Map<String, List<String>>
        fun bytes(): ByteArray
        fun close()
    }

    class Request private constructor(
        val url: String,
        val header: Map<String, String>,
        val method: HttpMethod,
        val mediaType: String,
        val payload: ByteArray,
        val waitTimeout: Long,
        val waitTimeoutUnit: TimeUnit
    ) {

        class Builder {
            private var header: Map<String, String> = emptyMap()
            private var url = ""
            private var method = HttpMethod.POST
            private var mediaType = ""
            private var payload = ByteArray(0)
            private var waitTimeout: Long = 0
            private var waitTimeoutUnit: TimeUnit = TimeUnit.MILLISECONDS

            fun addHeader(header: Map<String, String>): Builder {
                this.header = header
                return this
            }

            fun url(url: String): Builder {
                this.url = url
                return this
            }

            fun get(): Builder {
                this.method = HttpMethod.GET
                return this
            }

            fun post(mediaType: String, payload: ByteArray): Builder {
                this.method = HttpMethod.POST
                this.mediaType = mediaType
                this.payload = payload
                return this
            }

            fun waitTimeout(timeout: Long, timeUnit: TimeUnit): Builder {
                this.waitTimeout = timeout
                this.waitTimeoutUnit = timeUnit
                return this
            }

            fun build(): Request {
                return Request(
                    url,
                    header,
                    method,
                    mediaType,
                    payload,
                    waitTimeout,
                    waitTimeoutUnit
                )
            }


        }

    }

    interface IBuilder {

        fun connectTimeout(connectTimeout: Long, timeUnit: TimeUnit): IBuilder

        fun pingInterval(pingInterval: Long): IBuilder

        fun connectionPool(
            maxIdleConnections: Int,
            keepAliveDuration: Long,
            timeUnit: TimeUnit
        ): IBuilder

        fun build(): INetworkClient
    }

    interface IWebSocket {
        fun close(code: Int, reason: String)
        fun send(data: ByteArray): Boolean
        fun queueSize(): Long
    }

    interface IWebSocketListener {
        fun onOpen()
        fun onMessage(bytes: ByteArray?)
        fun onClosed(
            code: Int, reason: String?
        )

        fun onFailure(t: Throwable?)
    }
}