package com.hummer.im.signal._internals.rpc;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.google.protobuf.ByteString;
import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.IMRPC;
import com.hummer.im._internals.proto.Signal;
import com.hummer.im.model.signal.Message;
import com.hummer.im.model.signal.SendingOptions;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.id.User;

/**
 * 发送可靠信令
 */
public class RPCSendP2PSignalMessage extends IMRPC<
        Signal.P2PSignalRequest,
        Signal.P2PSignalRequest.Builder,
        Signal.P2PSignalResponse> {

    private final Message mMessage;
    private final SendingOptions mSendingOptions;
    private final User mReceiver;
    private final RichCompletion completion;

    public RPCSendP2PSignalMessage(Message message,
                                   SendingOptions sendingOptions,
                                   User receiver,
                                   RichCompletion completion) {
        this.mMessage = message;
        this.mSendingOptions = sendingOptions;
        this.mReceiver = receiver;
        this.completion = completion;
    }

    @Override
    public void buildHummerRequest(@NonNull Signal.P2PSignalRequest.Builder builder) {
        builder.setFromUid(HMR.getMe().getId())
                .setToUid(mReceiver.getId())
                .setType(mMessage.getType())
                .setContent(ByteString.copyFrom(mMessage.getData()));

        if (mMessage.getAppExtras() != null && mMessage.getAppExtras().size() > 0) {
            builder.putAllExtensions(mMessage.getAppExtras());
        }
        String reliable = mSendingOptions.isUnreliable() ? "no" : "yes";
        Signal.SignalMessageOption.Builder optionBuilder = Signal.SignalMessageOption.newBuilder();
        optionBuilder.setReliable(reliable);

        builder.setOption(optionBuilder);
    }

    @Override
    public void handleHummerSuccess(@NonNull Signal.P2PSignalResponse res) {
        mMessage.setServerAcceptedTs(res.getTimestamp() / 1000);
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void handleHummerError(@Nullable Signal.P2PSignalResponse res, @NonNull Error error) {
        CompletionUtils.dispatchFailure(completion, error);
    }

    @Override
    public String getHummerFunction() {
        return "P2PSignal";
    }
}
