package com.hummer.im.signal._internals;

import android.support.annotation.NonNull;

import com.google.protobuf.ByteString;
import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.Objects;
import com.hummer.im._internals.chatsvc.ChatServiceImpl;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.mq.Source;
import com.hummer.im._internals.proto.Im;
import com.hummer.im._internals.proto.Push;
import com.hummer.im._internals.proto.Signal;
import com.hummer.im._internals.shared.DispatchQueue;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im.model.signal.Message;
import com.hummer.im.signal._internals.rpc.RPCQueryUserState;
import com.hummer.im.signal._internals.rpc.RPCSendP2PSignalMessage;
import com.hummer.im.model.signal.SendingOptions;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.completion.RichCompletionArg;
import com.hummer.im.model.id.User;
import com.hummer.im.service.Channel;
import com.hummer.im.service.MQService;
import com.hummer.im.signal.SignalService;

import java.util.HashSet;

import static com.hummer.im.service.MQService.FetchStrategy.IgnoreBefore;

public class SignalServiceImpl implements SignalService, ServiceProvider.Service {

    private final HashSet<Listener> mMessageListeners = new HashSet<>();

    private static final String IM_URI
            = "service_api_gateway/cim.proto.PushService.UnreliableIMPushMsg";

    private void registerChatServiceExtensions() {
        ChatServiceImpl.registerParserExtension(new ChatServiceImpl.ParserExtension() {
            @Override
            public String toString() {
                return "SignalServiceImpl";
            }

            @Override
            public com.hummer.im.model.chat.Message parseMessage(String service, String function, byte[] data) {
                String actualUri = service + '/' + function;

                if (Objects.equals(actualUri, IM_URI)) {
                    try {
                        final Push.UnreliableIMPushMsgRequest req = Push.UnreliableIMPushMsgRequest
                                .newBuilder()
                                .mergeFrom(data)
                                .build();

                        if (req.getEnvName() != null && !req.getEnvName().isEmpty()) {
                            HMRContext.Region remoteRegion = HMRContext.Region.make(req.getRegion() + "/" + req
                                    .getEnvType() + "/" + req.getEnvName());
                            String localEnv = HMRContext.region.toString();

                            String remoteEnv = remoteRegion.toString();

                            if (!Objects.equals(remoteEnv, localEnv)) {
                                // 收到不属于当前接入区域的消息，直接忽略

                                Log.i(TAG, Trace.once().method("parseMessage")
                                        .msg("Ignored parse of different env")
                                        .info("localEnv", localEnv)
                                        .info("remoteEnv", remoteEnv));

                                return null;
                            }
                        }

                        if (req.getMsg().getAction() != Im.Action.kNotifySignalMessage) {
                            Log.i(TAG, Trace.once().method("parseMessage")
                                    .msg("Unregonized action for unreliable message"));
                            return null;
                        }

                        parseSignalMessage(req.getMsg().getContent(), req.getMsg().getTimestamp());

                    } catch (Throwable e) {
                        Log.e(TAG, Trace.once("Failed parsing UnreliableIMPushMsgRequest").info("Exception", e));
                    }
                }

                return null;
            }

            @Override
            public com.hummer.im.model.chat.Message parseMessage(Im.Msg msg, Source fromSource) {
                try {
                    if (msg.getAction() == Im.Action.kNotifySignalMessage) { // 信令消息
                        parseSignalMessage(msg.getContent(), msg.getTimestamp(), fromSource);
                    }
                } catch (Throwable t) {
                    Log.e(TAG, Trace.once().method("parse")
                            .info("action", msg.getAction())
                            .info("msg", msg.getContent())
                            .info("exception", t.getMessage()));
                }

                return null;
            }
        });
    }

    private void parseSignalMessage(final ByteString data, long timestamp, Source fromSource) throws Throwable {
        // 过滤组消息
        if (fromSource.getMode() instanceof Source.Shared) {
            return;
        }

        parseSignalMessage(data, timestamp);
    }

    private void parseSignalMessage(final ByteString data, long timestamp) throws Throwable {
        Signal.SignalMessage req = Signal.SignalMessage
                .newBuilder()
                .mergeFrom(data)
                .build();

        Message message = new Message(req.getType(), req.getContent().toByteArray());
        message.setServerAcceptedTs(timestamp);
        message.setAppExtras(req.getExtensionsMap());

        notifyReceivedMessage(message, new User(req.getFromUid()));
    }

    @Override
    public void queryOnlineStatusForUser(@NonNull final User user,
                                         @NonNull final HMR.CompletionArg<Boolean> completion) {
        Log.i(TAG, Trace.once()
                .method("queryOnlineStatusForUser")
                .info("user", user.getId()));

        RichCompletionArg<Boolean> rCompletion = new RichCompletionArg<>(completion,
                "SignalService::queryOnlineStatusForUser");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(
                    rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode")
            );
            return;
        }

        HMR.getService(Channel.class).run(new RPCQueryUserState(user, rCompletion));
    }

    @Override
    public void send(@NonNull final Message message,
                     @NonNull final User toUser,
                     @NonNull SendingOptions options,
                     @NonNull final HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "SignalService::sendMessage");
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(
                    rCompletion,
                    new Error(Error.Code.BadUser, "User not login, or using anonymous mode")
            );
            return;
        }

        HMR.getService(Channel.class).run(new RPCSendP2PSignalMessage(message, options, toUser, rCompletion));
    }

    @Override
    public void addListener(Listener listener) {
        synchronized (mMessageListeners) {
            mMessageListeners.add(listener);
        }

        Log.i(TAG, Trace.once()
                .method("addMessageListener")
                .info("name", listener.getClass().getSimpleName())
                .info("size", mMessageListeners.size()));
    }

    @Override
    public void removeListener(Listener listener) {
        synchronized (mMessageListeners) {
            mMessageListeners.remove(listener);
        }

        Log.i(TAG, Trace.once()
                .method("removeMessageListener")
                .info("name", listener.getClass().getSimpleName())
                .info("size", mMessageListeners.size()));
    }

    private void notifyReceivedMessage(@NonNull final Message message, @NonNull final User fromUser) {
        Log.i(TAG, Trace.once()
                .method("notifyReceivedMessage")
                .info("message.type", message.getType())
                .info("message.ts", message.getServerAcceptedTs())
                .info("fromUser", fromUser.getId()));

        DispatchQueue.main.async("SignalService::notifyReceivedMessage", new Runnable() {
            @Override
            public void run() {

                synchronized (mMessageListeners) {
                    HashSet<Listener> copyListener
                            = (HashSet<Listener>) mMessageListeners.clone();
                    for (Listener l : copyListener) {
                        l.onReceivedMessage(message, fromUser);
                    }
                }
            }
        });
    }

    @Override
    public Class[] staticDependencies() {
        return new Class[0];
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{Channel.class};
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return new Class[0];
    }

    @Override
    public void initService() {
        registerChatServiceExtensions();
        HMR.getService(MQService.class).
                addSource(new Source(new Source.Private("SignalService/P2P", IgnoreBefore, 5 * 60 * 1000L)));
    }

    @Override
    public void openService(@NonNull RichCompletion completion) {
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void closeService() {

    }

    private static final String TAG = "SignalService";
}
