package com.hummer.im.db._internals;

import android.content.Context;
import android.database.sqlite.SQLiteDatabase;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im.db.DBService;
import com.hummer.im._internals.services.user.UserService;
import com.hummer.im._internals.shared.DispatchQueue;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.j256.ormlite.android.apptools.OrmLiteSqliteOpenHelper;
import com.j256.ormlite.dao.Dao;
import com.j256.ormlite.dao.DaoManager;
import com.j256.ormlite.support.ConnectionSource;
import com.j256.ormlite.table.DatabaseTableConfig;

import junit.framework.Assert;
import java.lang.ref.WeakReference;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

public final class DBServiceImpl implements DBService,
        DBService.DaoSet,
        ServiceProvider.Service {

    private static final String TAG = "DBService";

    @Override
    public Class[] staticDependencies() {
        return null;
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{ UserService.class };
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return null;
    }

    @Override
    public void initService() {
    }

    @Override
    public void openService(@NonNull final RichCompletion completion) {
        // 命名格式：hmr_<appId>_<envType>_<envName>_<uid>
        HMRContext.work.async("openService::DBService", new Runnable() {
            @Override
            public void run() {
                if (!cachedDao.isEmpty()) {
                    Log.e("DBServiceImpl", Trace.once().method("openService")
                            .info("资源清理异常", null));
                }

                final String dbName = String.format(Locale.US, "hmr_%d_%s_%s_%d.db",
                        HMRContext.appId,
                        HMRContext.region.type,
                        HMRContext.region.name,
                        HMR.getMe().getId());

                Log.i(TAG, Trace.once().method("openService")
                        .info("VERSION", DB_VERSION)
                        .info("DB", dbName));

                dbHelper = new XDBHelper(HMRContext.getAppContext(), dbName, DB_VERSION, completion);
                dbHelper.getWritableDatabase();
            }
        });
    }

    @Override
    public void closeService() {
        cachedDao.clear();

        if (dbHelper != null) {
            dbHelper.close();
            dbHelper = null;  // 强制GCD尽早释放dbHelper资源
        }
    }

    @Override
    public <T> Dao<T, ?> create(DatabaseTableConfig<T> tableConfig, Class<T> modelClass) throws SQLException {
        String daoName = daoName(tableConfig, modelClass);

        //noinspection unchecked
        Dao<T, ?> dao = (Dao<T, ?>) cachedDao.get(daoName);

        if (dao == null) {
            // !!! 非常重要，在实际使用时发现，DaoManager在管理dao缓存时，table名称不会作为
            // 命名依据，这意味着如果同一个model，在进行分表存储时，可能内容会混乱。而且DaoManager
            // 内部的daoCache会在除了显示调用createDao之外的地方(例如TableUtils.dropTable)也被缓存。
            // 看起来，这是Ormlite的一个bug，暂时没有找到更好的解决方案，这里只是简单地清除掉所有cache，
            // 可以粗暴地避免出现上述问题。
            DaoManager.clearDaoCache();

            if (tableConfig != null && tableConfig.getTableName() != null) {
                dao = DaoManager.createDao(dbHelper.getConnectionSource(), tableConfig);
            } else {
                dao = DaoManager.createDao(dbHelper.getConnectionSource(), modelClass);
            }

            DaoManager.clearDaoCache();

            if (dao == null) {
                Log.e("DBServiceImpl", Trace.once().method("create")
                        .info("dao", null));
            }

            cachedDao.put(daoName, dao);
        }

        return dao;
    }

    @Override
    public <T> void remove(DatabaseTableConfig<T> tableConfig, Class<T> modelClass) {
        cachedDao.remove(daoName(tableConfig, modelClass));
    }

    private <T> String daoName(DatabaseTableConfig<T> tableConfig, Class<T> modelClass) {
        String daoName;

        if (tableConfig != null && tableConfig.getTableName() != null) {
            daoName = "table|" + tableConfig.getTableName();
        } else {
            Assert.assertNotNull("如果没有明确的表名，则应提供可以确定表名的Bean类型", modelClass);
            daoName = "model|" + modelClass.getName();
        }

        return daoName;
    }

    @Override
    public <Act extends Action> void execute(@NonNull Act act) {
        execute(act, null);
    }

    @Override
    public <Act extends Action> void execute(@NonNull final Act act, @Nullable final RichCompletion completion) {
        dbQueue.async(act.toString(), new Runnable() {
            @Override
            public void run() {
                try {
                    Log.i(TAG, Trace.once().method("execute").msg(act));

                    act.process(dbHelper, DBServiceImpl.this);

                    CompletionUtils.dispatchSuccess(completion);
                } catch (BreakByGuard breakByGuard) {
                    // 严格上来说，SQLExceptionBreakIntentionally 并不算是异常，所以不需要打印错误日志
                    if (breakByGuard.err == null) {
                        CompletionUtils.dispatchSuccess(completion);
                    } else {
                        CompletionUtils.dispatchFailure(completion, breakByGuard.err);
                    }
                } catch (SQLException exception) {
                    Error error = new Error(Error.Code.IOError, "数据库访问异常", exception);

                    Log.e(TAG, error, Trace.once().method("execute")
                            .msg(exception.getCause().toString()));

                    CompletionUtils.dispatchFailure(completion, error);
                } catch (Throwable t) {
                    Error error;
                    if (HMR.getMe() == null) {
                        error = new Error(Error.Code.BadUser, "Missing login user");
                    } else {
                        error = new Error(
                                Error.Code.UndefinedExceptions,
                                "Unknown exception: " + t.getLocalizedMessage()
                        );
                    }
                    CompletionUtils.dispatchFailure(completion, error);
                }
            }
        });
    }

    @Override
    public int getOldVersion() {
        return dbHelper.oldVersion;
    }

    @Override
    public int getNewVersion() {
        return dbHelper.newVersion;
    }

    @SuppressWarnings("FieldCanBeLocal")
    private static int DB_VERSION = 2;

    private XDBHelper dbHelper;
    private static final DispatchQueue dbQueue = new DispatchQueue(new DispatchQueue.WorkerHandler("hmr_db"));
    private static final Map<String, Dao> cachedDao = new HashMap<>();

    private static class XDBHelper extends OrmLiteSqliteOpenHelper {
        XDBHelper(Context context, String dbName, int dbVersion, RichCompletion completion) {
            super(context, dbName, null, dbVersion);

            this.completion = new WeakReference<>(completion);
        }

        @Override
        public void onCreate(SQLiteDatabase db, ConnectionSource connectionSource) {
            Log.i(TAG, Trace.once().method("onCreate"));
        }

        @Override
        public void onUpgrade(SQLiteDatabase db, ConnectionSource connectionSource, int oldVersion, int newVersion) {
            Log.i(TAG, Trace.once().method("onUpgrade"));
            this.oldVersion = oldVersion;
            this.newVersion = newVersion;
        }

        @Override
        public void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            Log.i(TAG, Trace.once().method("onDowngrade")
                    .info("oldVersion", oldVersion)
                    .info("newVersion", newVersion));
        }

        @Override
        public void onOpen(SQLiteDatabase db) {
            Log.i(TAG, Trace.once().method("onOpen")
                    .info("isOpen", db.isOpen())
                    .info("readonly", db.isReadOnly()));

            if (db.isOpen() && !db.isReadOnly()) {
                CompletionUtils.dispatchSuccess(completion.get());
            } else {
                CompletionUtils.dispatchFailure(completion.get(), new Error(
                        Error.Code.IOError, "Couldn't open database"));
            }
        }

        private final WeakReference<RichCompletion> completion;
        protected int oldVersion;
        protected int newVersion;
    }
}
