package com.hummer.im.db;

import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.hummer.im.Error;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.j256.ormlite.android.apptools.OrmLiteSqliteOpenHelper;
import com.j256.ormlite.dao.Dao;
import com.j256.ormlite.dao.DaoManager;
import com.j256.ormlite.table.DatabaseTableConfig;
import com.j256.ormlite.table.TableUtils;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

@SuppressWarnings({"unused", "unchecked"})
public final class DBActions implements DBService.Action {

    public <T> DBActions create(@NonNull final T model, final DatabaseTableConfig<T> tableConfig) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                daoSet.create(tableConfig, (Class<T>) model.getClass()).create(model);
            }

            @Override
            public String toString() {
                String string = "Create | model: " + model;

                if (tableConfig != null) {
                    string += ", table: " + tableConfig.getTableName();
                }

                return string;
            }
        });

        return this;
    }

    public <T> DBActions createIfNotExists(@NonNull final T model, final DatabaseTableConfig<T> tableConfig) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet)
                    throws SQLException {
                daoSet.create(tableConfig, (Class<T>) model.getClass()).createIfNotExists(model);
            }

            @Override
            public String toString() {
                String string = "createIfNotExists | model: " + model;

                if (tableConfig != null) {
                    string += ", table: " + tableConfig.getTableName();
                }

                return string;
            }
        });

        return this;
    }

    public <T> DBActions delete(@NonNull final T model, final DatabaseTableConfig<T> tableConfig) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                daoSet.create(tableConfig, (Class<T>) model.getClass()).delete(model);
            }

            @Override
            public String toString() {
                String string = "Delete | model: " + model;

                if (tableConfig != null) {
                    string += ", table: " + tableConfig.getTableName();
                }

                return string;
            }
        });

        return this;
    }

    public <T, I> DBActions deleteById(@NonNull final Class<T> modelClass, @NonNull final I id) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                Dao<T, I> dao = (Dao<T, I>) daoSet.create(null, modelClass);
                dao.deleteById(id);
            }

            @Override
            public String toString() {
                return "DeleteById | modelClass: " + modelClass.getSimpleName() + ", id: " + id;
            }
        });

        return this;
    }

    public <T, I> DBActions deleteById(@NonNull final DatabaseTableConfig<T> tableConfig, @NonNull final I id) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                Dao<T, I> dao = (Dao<T, I>) daoSet.create(tableConfig, null);
                dao.deleteById(id);
            }

            @Override
            public String toString() {
                return "DeleteById | table: " + tableConfig.getTableName() + ", id: " + id;
            }
        });

        return this;
    }

    public <T> DBActions deleteAll(@NonNull final Class<T> modelClass) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                TableUtils.clearTable(helper.getConnectionSource(), modelClass);
            }

            @Override
            public String toString() {
                return "DeleteAll | modelClass: " + modelClass.getSimpleName();
            }
        });

        return this;
    }

    public <T> DBActions deleteAll(@NonNull final DatabaseTableConfig<T> tableConfig) {
        if (tableConfig.getTableName() == null) {
            Log.e("DBActions", Trace.once().method("deleteAll")
                    .info("使用tableConfigs进行数据清除时，必须为其指定数据表名", null));
        }

        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                TableUtils.clearTable(helper.getConnectionSource(), tableConfig);
            }

            @Override
            public String toString() {
                return "DeleteAll | table: " + tableConfig.getTableName();
            }
        });

        return this;
    }

    public interface QueryAcceptor<R> {
        void onQueryResults(List<R> result);
    }

    public interface QueryOneAcceptor<R> {
        void onQueryResult(R result);
    }

    public <T> DBActions queryAll(@NonNull final Class<T> modelClass, @NonNull final QueryAcceptor acceptor) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                List<T> results = daoSet.create(null, modelClass).queryForAll();
                acceptor.onQueryResults(results);
            }

            @Override
            public String toString() {
                return "QueryAll | modelClass: " + modelClass.getSimpleName();
            }
        });

        return this;
    }

    public <T> DBActions queryAll(@NonNull final DatabaseTableConfig<T> tableConfig,
                                  @NonNull final QueryAcceptor acceptor) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                List<T> results = daoSet.create(tableConfig, null).queryForAll();
                acceptor.onQueryResults(results);
            }

            @Override
            public String toString() {
                return "QueryAll | table: " + tableConfig;
            }
        });

        return this;
    }

    public <T, ID> DBActions queryById(@NonNull final Class<T> modelClass,
                                       @NonNull final ID id,
                                       @NonNull final QueryOneAcceptor acceptor) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                Dao<T, ID> dao = (Dao<T, ID>) daoSet.create(null, modelClass);
                acceptor.onQueryResult(dao.queryForId(id));
            }

            @Override
            public String toString() {
                return "QueryById | modelClass: " + modelClass.getSimpleName() + ", id: " + id;
            }
        });

        return this;
    }

    public <T, ID> DBActions queryById(@NonNull final DatabaseTableConfig<T> tableConfig,
                                       @NonNull final ID id,
                                       @NonNull final QueryOneAcceptor acceptor) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                Dao<T, ID> dao = (Dao<T, ID>) daoSet.create(tableConfig, null);
                acceptor.onQueryResult(dao.queryForId(id));
            }

            @Override
            public String toString() {
                return "QueryById | table: " + tableConfig.getTableName() + ", id: " + id;
            }
        });

        return this;
    }
    
    public <T> DBActions update(@NonNull final T model, final DatabaseTableConfig<T> tableConfig) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                daoSet.create(tableConfig, (Class<T>) model.getClass()).update(model);
            }

            @Override
            public String toString() {
                String string = "Update | model: " + model;

                if (tableConfig != null) {
                    string += ", table: " + tableConfig.getTableName();
                }

                return string;
            }
        });

        return this;
    }

    public <T> DBActions createTableIfNeeded(@NonNull final Class<T> modelClass) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                TableUtils.createTableIfNotExists(helper.getConnectionSource(), modelClass);
                DaoManager.clearDaoCache();

                daoSet.create(null, modelClass);
            }

            @Override
            public String toString() {
                return "CreateTableIfNeeded | modelClass: " + modelClass.getSimpleName();
            }
        });

        return this;
    }

    public <T> DBActions createTableIfNeeded(@NonNull final DatabaseTableConfig<T> tableConfig) {
        if (tableConfig.getTableName() == null) {
            Log.e("DBActions", Trace.once().method("createTableIfNeeded")
                    .info("使用tableConfigs新建数据表时，必须为其指定数据表名", null));
        }

        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                TableUtils.createTableIfNotExists(helper.getConnectionSource(), tableConfig);
                DaoManager.clearDaoCache();

                daoSet.create(tableConfig, null);
            }

            @Override
            public String toString() {
                return "CreateTableIfNeeded | table: " + tableConfig.getTableName();
            }
        });

        return this;
    }

    public <T> DBActions dropTableIfExist(@NonNull final Class<T> modelClass) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                TableUtils.dropTable(helper.getConnectionSource(), modelClass, true);
                DaoManager.clearDaoCache();

                daoSet.remove(null, modelClass);
            }

            @Override
            public String toString() {
                return "DropTableIfExist | modelClass: " + modelClass.getSimpleName();
            }
        });

        return this;
    }

    public <T> DBActions dropTableIfExist(@NonNull final DatabaseTableConfig<T> tableConfig) {
        if (tableConfig.getTableName() == null) {
            Log.e("DBActions", Trace.once().method("dropTableIfExist")
                    .info("使用tableConfigs移除数据表时，必须为其指定数据表名", null));
        }
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                TableUtils.dropTable(helper.getConnectionSource(), tableConfig, true);
                DaoManager.clearDaoCache();

                daoSet.remove(tableConfig, null);
            }

            @Override
            public String toString() {
                return "DropTableIfExist | table: " + tableConfig.getTableName();
            }
        });

        return this;
    }

    public DBActions custom(@NonNull DBService.Action custom) {
        commands.add(custom);
        return this;
    }

    public interface Guarder {
        boolean shouldBreak();
    }

    public DBActions guard(final Error errIfBreak, @NonNull final Guarder guarder) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
                if (guarder.shouldBreak()) {
                    throw new DBService.BreakByGuard(errIfBreak);
                }
            }

            @Override
            public String toString() {
                return "Guard";
            }
        });

        return this;
    }

    public DBActions run(@NonNull final Runnable action) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) {
                action.run();
            }

            @Override
            public String toString() {
                return "Custom";
            }
        });
        return this;
    }

    public DBActions run(final String name, @NonNull final Runnable action) {
        commands.add(new DBService.Action() {
            @Override
            public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) {
                action.run();
            }

            @Override
            public String toString() {
                return name;
            }
        });
        return this;
    }

    @Override
    public void process(final OrmLiteSqliteOpenHelper helper, final DBService.DaoSet daoSet) throws SQLException {
        for (final DBService.Action cmd : commands) {
            cmd.process(helper, daoSet);
        }
    }

    @Override
    public String toString() {
        if (commands.size() == 1) {
            return commands.get(0).toString();
        } else {
            List<String> names = new ArrayList<>();

            for (DBService.Action act : commands) {
                names.add("{" + act.toString() + "}");
            }

            return "DBActions{" + TextUtils.join(", ", names) + '}';
        }
    }

    private List<DBService.Action> commands = new ArrayList<>();
}
