package com.hummer.im.model.completion;

import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.shared.DispatchQueue;

public final class RichCompletionArg<T> {

    public RichCompletionArg(@NonNull String description) {
        Looper looper = Looper.myLooper();
        if (looper == Looper.getMainLooper()) {
            this.dispatchQueue = new DispatchQueue(new DispatchQueue.LooperHandlerProvider(Looper.getMainLooper()));
            Log.i(TAG, Trace.once().method("CompletionArgInit")
                    .info("main dispatchQueue", this.dispatchQueue));
        } else if (looper == HMRContext.work.getHandler().getLooper()) {
            // 回到work队列
            this.dispatchQueue = HMRContext.work;
            Log.i(TAG, Trace.once().method("CompletionArgInit")
                    .info("work dispatchQueue", this.dispatchQueue));
        } else {
            // 默认使用Direct分派
            this.dispatchQueue = DispatchQueue.direct;
            Log.i(TAG, Trace.once().method("CompletionArgInit")
                    .info("internal dispatchQueue", this.dispatchQueue));
        }

        this.description = DesPrefix + description;
        HMRContext.recorder.start("[async]" + this.description);
    }

    public RichCompletionArg(final HMR.CompletionArg<T> completion, @NonNull String description) {
        this(description);
        onSuccess(new OnSuccessArg<T>() {
            @Override
            public void onSuccess(T result) {
                if (completion != null) {
                    completion.onSuccess(result);
                }
            }
        }).onFailure(new OnFailure() {
            @Override
            public void onFailure(Error error) {
                if (completion != null) {
                    completion.onFailed(error);
                }
            }
        });
    }

    public RichCompletionArg(@NonNull Handler handler, @NonNull String description) {
        this.dispatchQueue = new DispatchQueue(new DispatchQueue.LooperHandlerProvider(handler.getLooper()));
        this.description = DesPrefix + description;
        HMRContext.recorder.start("[async]" + this.description);
        Log.i(TAG, Trace.once().method("CompletionArgInit")
                .info("custom dispatchQueue with handler", this.dispatchQueue));
    }

    public RichCompletionArg<T> onSuccess(@NonNull OnSuccessArg<T> success) {
        this.successHandler = success;
        return this;
    }

    public RichCompletionArg<T> onFailure(@NonNull OnFailure failure) {
        this.failureHandler = failure;
        return this;
    }

    public String getDescription() {
        return this.description;
    }

    void dispatchSuccess(@NonNull T result) {
        Log.d(TAG, Trace.once().method("dispatchSuccess")
                .info("dispatchQueue", this.dispatchQueue));

        final T finalResult = result;
        dispatchQueue.async(this.description, new Runnable() {
            @Override
            public void run() {
                Log.i(TAG, Trace.once().method("dispatchSuccess")
                        .msg("dispatchQueue async, run"));
                if (successHandler != null) {
                    successHandler.onSuccess(finalResult);
                } else {
                    Log.i(TAG, Trace.once().method("dispatchSuccess")
                            .msg("dispatchQueue async, success handler = null"));
                }
            }
        });
    }

    void dispatchFailure(@NonNull Error error) {
        Log.i(TAG, Trace.once().method("dispatchFailure")
                .info("result", error.toString())
                .info("dispatchQueue", this.dispatchQueue));

        final Error finalError = error;
        dispatchQueue.async(this.description, new Runnable() {
            @Override
            public void run() {
                if (failureHandler != null) {
                    Log.i(TAG, Trace.once().method("dispatchFailure")
                            .info("result", finalError.toString()));
                    failureHandler.onFailure(finalError);
                } else {
                    Log.i(TAG, Trace.once().method("dispatchFailure")
                            .msg("dispatchQueue async, failure handler = null"));
                }
            }
        });
    }

    private static final String TAG = "RichCompletionArg";
    private static final String DesPrefix = "RichCompletionArg::";
    private final DispatchQueue dispatchQueue;
    private OnFailure failureHandler;
    private OnSuccessArg<T> successHandler;
    private String description;
}
