package com.hummer.im.model.chat.contents;

import android.graphics.BitmapFactory;
import android.support.annotation.NonNull;
import android.util.Patterns;
import android.webkit.URLUtil;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.proto.Im;
import com.hummer.im._internals.services.upload.Uploader;
import com.hummer.im._internals.services.upload.YYaliOSS.AliOSS;
import com.hummer.im._internals.shared.DispatchQueue;
import com.hummer.im._internals.shared.FileUtils;
import com.hummer.im._internals.shared.StringChain;
import com.hummer.im._internals.shared.statis.Util;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.chat.states.Preparing;
import com.hummer.im.service.UploadService;

import org.json.JSONObject;

/**
 * 图片消息内容
 */
public final class Image extends Content implements Content.Preparable {

    // Size 类型需要API 21，但是SDK的设计原则是尽可能使用低版本的SDK，因此只能退化为独立保存各属性
    public final int width;
    public final int height;
    public String originUrl;
    public String thumbUrl;

    private static final String TAG = "Image";

    /**
     * Image的工厂构造方法，用于创建图片消息内容实例
     *
     * @param path 本地图片文件路径
     * @return 如果文件存在，且是一个图片，则返回Image对象实例，否则返回null
     */
    public static Image createImage(@NonNull String path) {

        if (Patterns.WEB_URL.matcher(path).matches() || URLUtil.isValidUrl(path)) {
            Log.w(TAG, Trace.once().method("createImage")
                    .msg("Create url Image")
                    .info("path", path));
            return new Image(path, 0, 0);
        }

        if (!FileUtils.exists(path)) {
            Log.w(TAG, Trace.once().method("createImage")
                    .msg("Image not exists")
                    .info("path", path));

            return null;
        }

        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true; // 避免全量解码耗费资源

        BitmapFactory.decodeFile(path, options);
        if (options.outWidth == 0 || options.outHeight == 0) {
            Log.w(TAG, Trace.once().method("createImage").msg("Not an image"));
            return null;
        }

        return new Image(path, options.outWidth, options.outHeight);
    }

    @Override
    public void prepare(final PreparingCallback callback) {
        // 如果已经是网络图片，例如表情图片，则无需上传，直接回调成功即可
        if (isWebImageUrl()) {
            callback.onPrepareSuccess();
            return;
        }

        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true; // 避免全量解码耗费资源

        BitmapFactory.decodeFile(originUrl, options);
        if (options.outWidth == 0 || options.outHeight == 0) {
            Log.w(TAG, Trace.once().method("createImage").msg("Not an image"));

            DispatchQueue.main.async("Image::createImageFail", new Runnable() {
                @Override
                public void run() {
                    callback.onPrepareFailed(new Error(
                            Error.Code.InvalidParameters,
                            "Not an image"
                    ));
                }
            });

            return;
        }

        final BitmapFactory.Options finalOption = options;
        HMR.getService(UploadService.class).uploadFile(originUrl,
                new Uploader.UploadCallback<String>() {
                    @Override
                    public void onSuccess(String destUrl) {
                        originUrl = destUrl;
                        UploadService service = HMR.getService(UploadService.class);
                        if (service.getUploader() instanceof AliOSS) {
                            int length = 300;
                            String tbUrl = null;
                            if (finalOption.outWidth > length || finalOption.outHeight > length) {
                                tbUrl = service.getUploader().acquireThumbImageUrl(destUrl, length, length);
                            }

                            if (tbUrl == null) {
                                tbUrl = destUrl;
                            }

                            thumbUrl  = tbUrl;
                        }
                        callback.onPrepareSuccess();
                    }

                    @Override
                    public void onProgress(final float value) {
                        callback.onPrepare(new Preparing.ScalarProgress(value));
                    }

                    @Override
                    public void onFailure(Error err) {
                        callback.onPrepareFailed(err);
                    }
                });
    }

    @Override
    public String toString() {
        return "Image{" + new StringChain().acceptNullElements()
                .add("size",  String.valueOf(width) + "x" + String.valueOf(height))
                .add("origin", originUrl)
                .add("thumb",  thumbUrl)
                .toString();
    }

    private Image(String path, int width, int height) {
        this.originUrl = path;
        this.width     = width;
        this.height    = height;
    }

    private boolean isWebImageUrl() {
        return Patterns.WEB_URL.matcher(originUrl).matches() || URLUtil.isValidUrl(originUrl);
    }

    public static void registerCodecs() {
        registerCodec(new Codec() {
            @Override
            public int type() {
                return Im.MsgType.kImage_VALUE;
            }

            @Override
            public Class<? extends Content> contentClass() {
                return Image.class;
            }

            @Override
            public byte[] makePBBytes(Content content) {
                Image image = (Image) content;
                Im.ImageMsg.Builder builder = Im.ImageMsg.newBuilder();

                builder.setOriginalHeight(image.height);
                builder.setOriginalWidth(image.width);
                builder.setOriginalUrl(image.originUrl);

                if (!Util.empty(image.thumbUrl)) {
                    builder.setThumbnailUrl(image.thumbUrl);
                }

                return builder.build().toByteArray();
            }

            @Override
            public Content makeChatContent(byte[] data) throws Throwable {
                Im.ImageMsg msg = Im.ImageMsg.newBuilder().mergeFrom(data).build();
                Image img = new Image(
                        msg.getOriginalUrl(),
                        msg.getOriginalWidth(),
                        msg.getOriginalHeight()
                );

                if (!Util.empty(msg.getThumbnailUrl())) {
                    img.thumbUrl = msg.getThumbnailUrl();
                }

                return img;
            }

            static final String KeyWidth    = "width";
            static final String KeyHeight   = "height";
            static final String KeyOrigin   = "origin";
            static final String KeyThumb    = "thumb";

            @Override
            public String makeDBString(Content content) throws Throwable {
                Image image = (Image) content;

                JSONObject json = new JSONObject();
                json.put(KeyWidth,    image.width);
                json.put(KeyHeight,   image.height);
                json.put(KeyOrigin,   image.originUrl);

                if (!Util.empty(image.thumbUrl)) {
                    json.put(KeyThumb,    image.thumbUrl);
                }
                return json.toString();
            }

            @Override
            public Content makeChatContent(String data) throws Throwable {
                JSONObject json = new JSONObject(data);

                Image image = new Image(
                        json.getString(KeyOrigin),
                        json.getInt(KeyWidth),
                        json.getInt(KeyHeight)
                );

                if (json.has(KeyThumb)) {
                    image.thumbUrl = json.getString(KeyThumb);
                }

                return image;
            }

            @Override
            public String toString() {
                return "ImageCodec";
            }
        });
    }
}
