package com.hummer.im.model.chat.contents;

import android.media.MediaMetadataRetriever;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Patterns;
import android.webkit.URLUtil;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.proto.Im;
import com.hummer.im._internals.services.upload.Uploader;
import com.hummer.im._internals.shared.FileUtils;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.chat.states.Preparing;
import com.hummer.im.service.UploadService;

import org.json.JSONObject;

import java.util.Locale;

/**
 * 语音消息类型
 */
public final class Audio extends Content implements Content.Preparable {

    /**
     * 语音的资源定位路径，如果Audio实例尚未发送成功，通常为本地路径，否则为远端云存储的资源路径
     */
    public String url;

    /**
     * 语音的持续时长，单位为毫秒(ms)
     */
    public int duration;

    /**
     * 通过一个语音路径来创建一个Audio语音消息类型实例
     *
     * @param path 语音资源的路径
     * @return 如果该资源存在，且可以被正确解析为语音资源，则返回一个语音消息内容的实例，否则返回null
     */
    public static Audio create(@NonNull String path) {
        if (!FileUtils.exists(path)) {
            Log.w(TAG, Trace.once().method("create")
                    .msg("File not exists")
                    .info("path", path));

            return null;
        }

        int duration = 0;

        MediaMetadataRetriever retriever = new MediaMetadataRetriever();
        retriever.setDataSource(path);
        String durationString = retriever.extractMetadata(MediaMetadataRetriever.METADATA_KEY_DURATION);
        retriever.release(); // 记得要释放
        if (durationString != null && !TextUtils.isEmpty(durationString)) {
            duration = Integer.parseInt(durationString);
        }

        if (duration == 0) {
            // 无效音频
            return null;
        }

        return new Audio(path, duration);
    }

    /**
     * 语音消息内容的编码器、解码器注册方法
     */
    public static void registerCodecs() {
        registerCodec(new Codec() {
            @Override
            public int type() {
                return Im.MsgType.kAudio_VALUE;
            }

            @Override
            public Class<? extends Content> contentClass() {
                return Audio.class;
            }

            @Override
            public byte[] makePBBytes(Content content) {
                Audio audio = (Audio) content;
                Im.AudioMsg.Builder builder = Im.AudioMsg.newBuilder();

                builder.setUrl(audio.url);
                builder.setDuration(audio.duration);

                return builder.build().toByteArray();
            }

            @Override
            public Content makeChatContent(byte[] data) throws Throwable {
                Im.AudioMsg msg = Im.AudioMsg.newBuilder().mergeFrom(data).build();
                return new Audio(msg.getUrl(), msg.getDuration());
            }

            private static final String KeyUrl      = "url";
            private static final String KeyDuration = "dur";

            @Override
            public String makeDBString(Content content) throws Throwable {
                Audio audio = (Audio) content;

                JSONObject json = new JSONObject();
                json.put(KeyUrl,      audio.url);
                json.put(KeyDuration, audio.duration);

                return json.toString();
            }

            @Override
            public Content makeChatContent(String data) throws Throwable {
                JSONObject json = new JSONObject(data);
                return new Audio(json.getString(KeyUrl), json.getInt(KeyDuration));
            }

            @Override
            public String toString() {
                return "AudioCodec";
            }
        });
    }


    @Override
    public String toString() {
        return String.format(Locale.US, "Audio{%dms, %s}", duration, url);
    }

    @Override
    public void prepare(final PreparingCallback callback) {
        if (url.isEmpty()) {
            Log.e("Audio", Trace.once().method("prepare")
                    .info("url", url));
            callback.onPrepareFailed(new Error(Error.Code.InvalidParameters, "Invalid url."));
            return;
        }

        if (Patterns.WEB_URL.matcher(url).matches() || URLUtil.isValidUrl(url)) {
            callback.onPrepareSuccess();
            return;
        }

        HMR.getService(UploadService.class).uploadFile(url, new Uploader.UploadCallback<String>() {
            @Override
            public void onSuccess(String urlString) {
                url = urlString;
                callback.onPrepareSuccess();
            }

            @Override
            public void onProgress(final float value) {
                callback.onPrepare(new Preparing.ScalarProgress(value));
            }

            @Override
            public void onFailure(Error err) {
                callback.onPrepareFailed(err);
            }
        });
    }

    private Audio(@NonNull String path, Integer duration) {
        this.url      = path;
        this.duration = duration;
    }

    private static final String TAG = "Audio";
}
