package com.hummer.im._internals.shared;

import android.text.TextUtils;

import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.shared.statis.MetricsHttpEncryptUtil;
import com.hummer.im._internals.shared.statis.MetricsWorker;
import com.hummer.im._internals.shared.statis.StatisHttpEncryptUtil;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.security.InvalidParameterException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

@SuppressWarnings({"WeakerAccess", "unused", "SameParameterValue"})
public final class HiidoReporter {

    public static final DispatchQueue reportQueue = new DispatchQueue(
            new DispatchQueue.WorkerHandler("hmr_report"));
    public static String TAG = "Reporter";

    public enum Region {
        China, Overseas,
    }

    public static class Field {

        public static Field from(final Integer intValue) {
            return new Field(String.valueOf(intValue));
        }

        public static Field from(final Long longValue) {
            return new Field(String.valueOf(longValue));
        }

        public static Field from(final String stringValue) {
            return new Field(stringValue);
        }

        @Override
        public String toString() {
            return value;
        }

        private Field(String value) {
            this.value = value;
        }

        private String value;
    }

    // FIXME: hago分支默认是海外？
    public static void setRegion(Region region) {
        switch (region) {
            case China:
                host = "ylog.hiido.com";
                //默认 "mlog.hiido.com"
                metricsHost = "klog.hiido.com";
                break;
            case Overseas:
                host = "hlog.hiido.com";
                //默认 "mlog.hiido.com"
                metricsHost = "klog.hiido.com";
                break;
            default:
                Log.e("HiidoReporter", Trace.once().method("setRegion")
                        .info("hiido.Reporter | Unknown region: ", region));
        }
        metricsWorker = new MetricsWorker("44b8f2b988e258e1d036acd459afaacc",
                new MetricsHttpEncryptUtil(metricsHost, null),
                new StatisHttpEncryptUtil(host, null));
    }

    /**
     * 旧的海度上报接口中，会区分不同类型的字段，所以这里提供了一个工厂方法以便于进行转换
     */
    public static Map<String, Field> makeFields(Map<String, Integer> intFields,
                                                Map<String, Long> longFields,
                                                Map<String, String> stringFields) {

        HashMap<String, Field> fields = new HashMap<>();

        for (Map.Entry<String, Integer> entry : intFields.entrySet()) {
            fields.put(entry.getKey(), HiidoReporter.Field.from(entry.getValue()));
        }
        for (Map.Entry<String, Long> entry : longFields.entrySet()) {
            fields.put(entry.getKey(), HiidoReporter.Field.from(entry.getValue()));
        }
        for (Map.Entry<String, String> entry : stringFields.entrySet()) {
            fields.put(entry.getKey(), HiidoReporter.Field.from(entry.getValue()));
        }

        return fields;
    }

    public static void report(final String act, final Map<String, Field> fields) {
        Log.i(TAG, "report | " + fields);

        new Thread(new Runnable() {
            @Override
            public void run() {
                HttpURLConnection conn = null;
                try {
                    String query = queryString(act, fields);
                    if (query == null) {
                        // 上报不是核心路径，应进行容错，而不是崩溃
                        return;
                    }

                    URL url = new URL(String.format(Locale.US, "https://%s/c.gif?%s",
                            HiidoReporter.host,
                            query));

                    conn = (HttpURLConnection) url.openConnection();
                    conn.setRequestMethod("GET");
                    conn.setReadTimeout(6000);
                    conn.setConnectTimeout(10000);

                    // 调用此方法就不必再使用conn.connect()方法
                    int responseCode = conn.getResponseCode();
                    if (responseCode != 200) {
                        String failure = getStringFromInputStream(conn.getErrorStream());
                        Log.w(TAG, "report | Failed: [" + responseCode + "] " + failure);
                    }
                } catch (Exception e) {
                    Log.e(TAG, "report | Exception: " + e.getLocalizedMessage());
                } finally {
                    if (conn != null) {
                        conn.disconnect();
                    }
                }
            }
        }).start();
    }

    public static void reportActTemporary(final String act,
                                          final Map<String, Integer> intFields,
                                          final Map<String, Long> longFields,
                                          final Map<String, String> stringFields) {
        reportQueue.async("HiidoReporter::reportActTemporary", new Runnable() {
            @Override
            public void run() {
                try {
                    metricsWorker.reportHiidoTemporary(act, intFields, longFields, stringFields);
                } catch (Throwable e) {
                    Log.e(TAG, String.format(Locale.US, "reportStatisticContentTemporary error:%s",
                            e.getLocalizedMessage()));
                }
            }
        });
    }

    public static void reportReturnCode(final int scode, final String uri,
                                        final long timeConsumption, final String code) {
        reportQueue.async("HiidoReporter::reportReturenCode", new Runnable() {
            @Override
            public void run() {
                try {
                    metricsWorker.reportReturnCode(scode, uri, timeConsumption, code);
                } catch (Throwable e) {
                    Log.e(TAG, String.format(Locale.US, "reportReturnCode error:%s",
                            e.getLocalizedMessage()));
                }
            }
        });
    }

    public static void reportReturnCodeTemporary(final int scode, final String uri,
                                                 final long timeConsumption, final String code) {
        reportQueue.async("HiidoReporter::reportReturnCodeTemporary", new Runnable() {
            @Override
            public void run() {
                try {
                    metricsWorker.reportReturnCodeTemporary(scode, uri, timeConsumption, code);
                } catch (Throwable e) {
                    Log.e(TAG, String.format(Locale.US, "reportReturnCodeTemporary error:%s",
                            e.getLocalizedMessage()));
                }
            }
        });
    }

    public static void reportReturnCodeTemporary(String funcName, long rtt, int code) {
        if (rtt < 0) {
            return;
        }

        String uri = funcName + "/" + String.valueOf(HMRContext.appId);
        HiidoReporter.reportReturnCodeTemporary(50216, uri, rtt, String.valueOf(code));
    }

    public static void reportCount(final int scode, final String uri, final String countName, final long count) {
        reportQueue.async("HiidoReporter::reportCount", new Runnable() {
            @Override
            public void run() {
                try {
                    metricsWorker.reportCount(scode, uri, countName, count);
                } catch (Throwable e) {
                    Log.e(TAG, String.format(Locale.US, "report count error:%s", e.getLocalizedMessage()));
                }
            }
        });
    }

    public static void reportCount(final int scode, final String uri, final String countName,
                                   final long count, final int times) {
        reportQueue.async("HiidoReporter::reportCountWithtimes", new Runnable() {
            @Override
            public void run() {
                try {
                    metricsWorker.reportCount(scode, uri, countName, count, times);
                } catch (Throwable e) {
                    Log.e(TAG, String.format(Locale.US, "reportCount error:%s",
                            e.getLocalizedMessage()));
                }
            }
        });
    }

    private static String queryString(String act, Map<String, Field> fields) throws InvalidParameterException {
        long ts = System.currentTimeMillis() / 1000;
        String key = actionKey(act, ts);
        if (key == null) {
            return null;
        }

        List<String> components = new ArrayList<>();
        components.add(String.format(Locale.US, "act=%s", act));
        components.add(String.format(Locale.US, "time=%d", ts));
        components.add(String.format(Locale.US, "key=%s", key));  // PC和移动端(Act)上报必须同时携带校验key

        for (Map.Entry<String, Field> entry : fields.entrySet()) {
            Field field = entry.getValue();
            if (!isAcceptableField(field)) {
                Log.w(TAG, Trace.once("Null Field: %s", entry.getKey()));
                continue;
            }

            try {
                String encodedValue = URLEncoder.encode(entry.getValue().toString(), "utf-8");
                components.add(String.format(Locale.US, "%s=%s", entry.getKey(), encodedValue));
            } catch (UnsupportedEncodingException e) {
                Log.w(TAG, Trace.once("Invalid Field: %s = %s", entry.getKey(), field));
            }
        }

        return TextUtils.join("&", components);
    }

    private static String getStringFromInputStream(InputStream is) throws IOException {
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        byte[] buffer = new byte[1024];
        int len;
        while ((len = is.read(buffer)) != -1) {
            os.write(buffer, 0, len);
        }
        is.close();
        String state = os.toString();
        os.close();

        return state;
    }

    public static String actionKey(String act, long ts) {
        if (TextUtils.isEmpty(act)) {
            Log.e("HiidoReporter", Trace.once().method("actionKey")
                    .info("act", act));
        }

        // 具体算法请参阅：
        // https://cloud.hiido.com/document/act

        String content = act + String.valueOf(ts) + "HiidoYYSystem";
        try {
            MessageDigest md5 = MessageDigest.getInstance("MD5");
            byte[] bytes = md5.digest(content.getBytes());
            StringBuilder sb = new StringBuilder();
            for (byte b : bytes) {
                String temp = Integer.toHexString(b & 0xff);
                if (temp.length() == 1) {
                    temp = "0" + temp;
                }
                sb.append(temp);
            }
            return sb.toString().toLowerCase();
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
            return null;
        }
    }

    private static String host;

    private static String metricsHost;

    private static MetricsWorker metricsWorker;

    static {
        setRegion(Region.Overseas);    // 目前主要服务于海外市场，因此默认使用海外域名
    }

    private static boolean isAcceptableField(Field field) {
        return field != null;
    }
}
