package com.hummer.im._internals.services.mq;

import android.os.Build;
import android.util.SparseIntArray;

import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im.model.id.User;
import com.hummer.im._internals.shared.HiidoReporter;

import java.util.HashMap;
import java.util.Map;

public final class Statistics {

    private static Map<Integer, String> codeNames = new HashMap<Integer, String>() {{
        put(Codes.ExceptionalDispatch, "ExceptionalDispatch");
        put(Codes.NotResetIsDraining,  "NotResetIsDraining");
        put(Codes.ImpossibleScene,     "ImpossibleScene");
    }};

    public static void report(final int errCode, final Fields fields) {
        final String codeName = codeNames.get(errCode);
        if (codeName == null) {
            // 忽略错误（无法识别）的上报内容
            return;
        }

        int frequency = Frequencies.get(errCode, 0);
        boolean customizedFrequency = (frequency > 0);

        if (customizedFrequency) {
            boolean shouldReport = false;
            int count = RequestedCounts.get(errCode, 0);
            count += 1;

            if (count >= frequency) {
                shouldReport = true;
                count = 0;
            }

            RequestedCounts.put(errCode, count);

            if (!shouldReport) {
                // 仅仅是更新请求次数，不作上报
                return;
            }
        }

        Log.w("MessageService", Trace.once().method("reportException")
                .info("type",   codeName)
                .info("fields", fields.toString()));

        HashMap<String, HiidoReporter.Field> fieldsMap = new HashMap<String, HiidoReporter.Field>() {{
            // 固有字段（必备）
            put("errCode",       HiidoReporter.Field.from(errCode));
            put("errType",       HiidoReporter.Field.from(codeName));
            put("appId",         HiidoReporter.Field.from(HMRContext.getAppId()));
            put("sys",           HiidoReporter.Field.from(2L)); // For Android
            put("operateSystem", HiidoReporter.Field.from(Build.VERSION.RELEASE));
            put("phoneModel",    HiidoReporter.Field.from(getDeviceName()));
            put("sdkver",        HiidoReporter.Field.from(HMR.getVersion()));

            // 可空字段
            put("logId",         HiidoReporter.Field.from(fields.mLogId));
            put("localSeqId",    HiidoReporter.Field.from(fields.localSeqId));
            put("maxSeqId",      HiidoReporter.Field.from(fields.maxSeqId));
            put("remoteSeqId",   HiidoReporter.Field.from(fields.remoteSeqId));
            put("errInfo",       HiidoReporter.Field.from(fields.errInfo));
        }};

        User me = HMR.getMe();
        if (me != null) {
            fieldsMap.put("uid", HiidoReporter.Field.from(me.getId()));
        }

        HiidoReporter.report("cim2", fieldsMap);
    }

    public static void report(final Fields fields) {
        HashMap<String, HiidoReporter.Field> fieldsMap = new HashMap<String, HiidoReporter.Field>() {{
            put("appId", HiidoReporter.Field.from(HMRContext.appId));
            put("sys",   HiidoReporter.Field.from(2L));        // For Android
        }};

        User me = HMR.getMe();
        if (me != null) {
            fieldsMap.put("uid", HiidoReporter.Field.from(me.getId()));
        }

        fieldsMap.put("localSeqId",  HiidoReporter.Field.from(fields.localSeqId));
        fieldsMap.put("maxSeqId",    HiidoReporter.Field.from(fields.maxSeqId));
        fieldsMap.put("remoteSeqId", HiidoReporter.Field.from(fields.remoteSeqId));
        fieldsMap.put("errCode",     HiidoReporter.Field.from(fields.errCode));
        fieldsMap.put("errInfo",     HiidoReporter.Field.from(fields.errInfo));

        if (fields.errCode == Codes.UndefinedError) {
            fieldsMap.put("errType", HiidoReporter.Field.from("UndefinedError"));
        } else if (fields.errCode >= Codes.PullMsgNull && fields.errCode <= Codes.ParseMsgErr) {
            String[] types = {"PullMsgNull", "LocalMaxRemote", "ContinuePullErr", "ParseMsgErr"};
            fieldsMap.put("errType", HiidoReporter.Field.from(types[fields.errCode - 1]));
        }

        HiidoReporter.report(act, fieldsMap);
    }

    private static final String act = "cim2";

    private static String getDeviceName() {
        final String manufacturer = Build.MANUFACTURER;
        final String model = Build.MODEL;
        return model.startsWith(manufacturer) ? capitalizedPhrase(model) :
                capitalizedPhrase(manufacturer) + " " + model;
    }

    private static String capitalizedPhrase(String s) {
        if (s == null || s.length() == 0) {
            return s;
        } else {
            StringBuilder phrase = new StringBuilder();
            boolean next = true;
            for (char c : s.toCharArray()) {
                if (next && Character.isLetter(c) || Character.isWhitespace(c)) {
                    next = Character.isWhitespace(c = Character.toUpperCase(c));
                }

                phrase.append(c);
            }
            return phrase.toString();
        }
    }

    @SuppressWarnings("WeakerAccess")
    public final static class Codes {
        /* 1 到 4 为已经废弃的错误类型，不再上报。但是为了前向兼容，不再使用它们 */

        public static final Integer PullMsgNull     = 1;
        public static final Integer LocalMaxRemote  = 2;
        public static final Integer ContinuePullErr = 3;
        public static final Integer ParseMsgErr     = 4;

        public static final Integer ExceptionalDispatch = 5;
        public static final Integer NotResetIsDraining  = 6;
        public static final Integer ImpossibleScene     = 7;

        public static final Integer UndefinedError  = 999;
    }

    public  static SparseIntArray Frequencies     = new SparseIntArray();
    private static SparseIntArray RequestedCounts = new SparseIntArray();

    public static class Fields {
        public int errCode;
        public Long mLogId;
        public Long localSeqId;
        public Long remoteSeqId;
        public Long maxSeqId;
        public String errInfo;

        public Fields(int errCode) {
            this.errCode = errCode;
        }

        public Fields() {

        }

        public void setLocalSeqId(Long localSeqId) {
            this.localSeqId = localSeqId;
        }

        public void setRemoteSeqId(Long remoteSeqId) {
            this.remoteSeqId = remoteSeqId;
        }

        public void setMaxSeqId(Long maxSeqId) {
            this.maxSeqId = maxSeqId;
        }

        public void setErrInfo(String errInfo) {
            this.errInfo = errInfo;
        }
    }
}

