package com.hummer.im._internals.roaming;

import android.support.annotation.NonNull;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.chatsvc.MessageParser;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.proto.History;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.store.FetchingClauses;
import com.hummer.im.model.chat.store.FetchingResult;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.OnFailure;
import com.hummer.im.model.completion.OnSuccessArg;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.completion.RichCompletionArg;
import com.hummer.im.model.id.Identifiable;
import com.hummer.im.service.Channel;
import com.hummer.im.service.RoamingService;

import java.util.ArrayList;
import java.util.List;

public class RoamingServiceImpl implements RoamingService, ServiceProvider.Service {

    private static final String TAG = "RoamingServiceImpl";

    @Override
    public Class[] staticDependencies() {
        return null;
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{ Channel.class };
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return null;
    }

    @Override
    public void initService() {

    }

    @Override
    public void openService(@NonNull RichCompletion completion) {
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void closeService() {

    }

    @Override
    public void fetchMessages(Identifiable target, FetchingClauses clauses,
                              final HMR.CompletionArg<FetchingResult> completion) {
        final RichCompletionArg rCompletion = new RichCompletionArg(completion, "RoamingService::fetchMessages");
        Error error = null;
        if (HMR.getState() != HMR.State.Opened || HMR.getMe().isAnonymous()) {
            error = new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode");
        }
        if (clauses == null) {
            error = new Error(Error.Code.InvalidParameters, "clauses should not be null");
        } else if (target == null) {
            error = new Error(Error.Code.InvalidParameters, "<target> should not be null");
        }

        if (error != null) {
            Log.e(TAG, Trace.once().method("fetchRambleMessages").msg(error.toString()));
            CompletionUtils.dispatchFailure(rCompletion, error);
            return;
        }

        if (clauses.getBeforeTimestamp() == null) {
            clauses.setBeforeTimestamp(0L);
        }

        int maxCount = 100;
        if (clauses.getLimit() == null || clauses.getLimit() > maxCount) {
            clauses.setLimit(maxCount);
        } else if (clauses.getLimit() < 1) {
            clauses.setLimit(1);
        }

        HMR.getService(Channel.class).run(new RPCFetchRemoteMessage(target,
                clauses,
                new RichCompletionArg<History.ReverseListChatHistoryResponse>("RoamingService::RPCFetchRemoteMessage")
                        .onSuccess(new OnSuccessArg<History.ReverseListChatHistoryResponse>() {
                            @Override
                            public void onSuccess(History.ReverseListChatHistoryResponse response) {
                                List<Message> messages = new ArrayList<>();
                                for (History.HistoryMsg msg : response.getMsgsList()) {
                                    Message chatMessage = RoamingServiceImpl.this.parseMsg(msg);
                                    if (chatMessage != null) {
                                        messages.add(RoamingServiceImpl.this.parseMsg(msg));
                                    }
                                }

                                FetchingResult result = new FetchingResult();
                                result.setHasMore(response.getHasMore());
                                result.setNextTimestamp(response.getNextTimestamp());
                                result.setMessages(messages);

                                CompletionUtils.dispatchSuccess(rCompletion, result);
                            }
                        })
                        .onFailure(new OnFailure() {
                            @Override
                            public void onFailure(Error err) {
                                CompletionUtils.dispatchFailure(rCompletion, err);
                            }
                        })));
    }

    private Message parseMsg(History.HistoryMsg msg) {
        return MessageParser.parseContent(msg.getContent(), msg.getTimestamp(), msg.getAction());
    }
}
