package com.hummer.im._internals.mq;

import android.content.SharedPreferences;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.Objects;
import com.hummer.im._internals.PrefStorage;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.proto.Push;
import com.hummer.im._internals.services.mq.RPCPullingResponse;
import com.hummer.im._internals.services.mq.Statistics;
import com.hummer.im.model.completion.RichCompletionArg;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.OnFailure;
import com.hummer.im.model.completion.OnSuccessArg;
import com.hummer.im.service.Channel;
import com.hummer.im.service.MQService;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Locale;

public final class Source implements MQService.Source {

    private final Mode mode;
    private Long seqId;
    private Long lastPullAt;
    private boolean isDraining;
    private boolean sourceChanged;

    public Source(Mode mode) {
        this.mode = mode;
    }

    public Mode getMode() {
        return mode;
    }

    @Override
    public void onTimerPulse(final MQService.MessagesDispatcher dispatcher) {
        HMRContext.work.async("Source::onTimerPulse:" + this.mode.sourceName(), new Runnable() {
            @Override
            public void run() {
                if (Source.this.isOverdue(TimeoutDuration)) {
                    if (isDraining) {
                        Statistics.report(Statistics.Codes.NotResetIsDraining, new Statistics.Fields());
                    }

                    // 进一步预防 http://ylbug.yypm.com:8081/browse/ANDYN-10515 问题可能带来的影响
                    // 防止rpc过程异常导致没有正确重置isDraining标识
                    isDraining = false;
                }

                Source.this.drainMessagesIfNeeded(dispatcher);
            }
        });
    }

    @Override
    public void onManualPullingRequest(final MQService.MessagesDispatcher dispatcher) {
        HMRContext.work.async("Source::onManualPullingRequest", new Runnable() {
            @Override
            public void run() {
                Source.this.drainMessagesIfNeeded(dispatcher);
            }
        });
    }

    @Override
    public void onNetworkReconnected(final MQService.MessagesDispatcher dispatcher) {
        HMRContext.work.async("Source::onNetworkReconnected", new Runnable() {
            @Override
            public void run() {
                // 网络断连时间无法确定，如果不重置lastDrainedAt的话会影响后台的消息拉取时延统计结果
                lastPullAt = null;
                Source.this.drainMessagesIfNeeded(dispatcher);
            }
        });
    }

    @Override
    public void start(final MQService.MessagesDispatcher dispatcher) {
        HMRContext.work.async("Source::start", new Runnable() {
            @Override
            public void run() {
                Source.this.performStarting(dispatcher);
            }
        });
    }

    @Override
    public void stop() {
        HMRContext.work.async("Source::stop", new Runnable() {
            @Override
            public void run() {
                performStopping();
            }
        });
    }

    @Override
    public String toString() {
        return mode.sourceName();
    }

    private void performStarting(final MQService.MessagesDispatcher dispatcher) {
        Log.i(mode.sourceName(), Trace.once().method("performStarting"));

        isDraining = false;
        lastPullAt = null;

        mode.start(new Runnable() {
            @Override
            public void run() {
                HMRContext.work.async(mode.sourceName(), new Runnable() {
                    @Override
                    public void run() {
                        sourceChanged = true;
                        Source.this.drainMessagesIfNeeded(dispatcher);
                    }
                });
            }
        });

        mode.loadSeqId(getStrategy(mode), new RichCompletionArg<Long>("Source::loadSeqId:" + mode.sourceName())
                .onSuccess(new OnSuccessArg<Long>() {
                    @Override
                    public void onSuccess(final Long loadedSeqId) {
                        if (loadedSeqId == null) {
                            Log.e("Source", Trace.once().method("performStarting").msg("loadedSeqId is <null>"));
                        }

                        HMRContext.work.async(mode.sourceName() + "::loadSeqIdSuccess", new Runnable() {
                            @Override
                            public void run() {
                                // 当本地没有seqId（例如重装app）时，应强制写入本地，避免下次启动再去拉取
                                Source.this.setSeqId(loadedSeqId, Source.this.seqId == null);
                                Source.this.drainMessagesIfNeeded(dispatcher);
                            }
                        });
                    }
                })
                .onFailure(new OnFailure() {
                    @Override
                    public void onFailure(Error err) {
                        HMRContext.work.async(mode.sourceName() + "::loadSeqIdFailure", new Runnable() {
                            @Override
                            public void run() {
                                // 如果失败，则无论如何都从0开始，避免因为该流程而导致始终无法拉取消息
                                // 缺点是，有可能增加会拉取到冗余、重复消息的概率
                                Source.this.setSeqId(0L, false);
                                Source.this.drainMessagesIfNeeded(dispatcher);
                            }
                        });
                    }
                }));
    }

    private MQService.FetchStrategy getStrategy(Mode mode) {
        if (mode.getStrategy() != null) {
            return mode.getStrategy();
        }

        if (HMR.getService(MQService.class).getFetchStrategy() != null) {
            return HMR.getService(MQService.class).getFetchStrategy();
        }

        return MQService.FetchStrategy.Continuously;

    }

    private void performStopping() {
        Log.i(mode.sourceName(), Trace.once().method("performStopping"));

        mode.stop();

        isDraining = false;
        lastPullAt = null;
        seqId = null;
    }

    private static final long TimeoutDuration = 60 * 1000;
    private static final long DEFAULT_PULLING_PERIOD = 15 * 60 * 1000; // 测试暂定1分钟

    private boolean isOverdue(long period) {
        long tolerance = period / 30;
        long now = System.currentTimeMillis();
        return (lastPullAt == null || (now - lastPullAt) >= (period - tolerance));
    }

    private void drainMessagesIfNeeded(final MQService.MessagesDispatcher dispatcher) {
        @SuppressWarnings("PointlessBooleanExpression")
        boolean shouldIgnore =
                // 当使用SkipFetched模式启动时，seqId可能存在短暂窗口处于不确定的状态。在这个窗口期间，如果因为超期
                // 检测而触发了drainMessagesIfNeeded调用，应直接忽略掉。
                seqId == null

                        // 另一个消息拉取尝试正在进行，忽略掉冗余拉取请求
                        || isDraining

                        // 既没有更多消息，也没有过期，忽略拉取请求。主要是因为时间脉冲触发的拉取尝试导致的。
                        || !(sourceChanged || isOverdue(mode.getPullingPeriod()));

        if (shouldIgnore) {
            return;
        }

        Log.i(mode.sourceName(), Trace.once().method("drainMessagesIfNeeded")
                .info("seqId", seqId)
                .info("changed", sourceChanged)
                .info("draining", isDraining)
                .info("overdue", isOverdue(mode.getPullingPeriod())));

        sourceChanged = false; // Reset dirty flag before draining
        isDraining = true;
        drainMessages(dispatcher, seqId, lastPullAt == null, new Runnable() {
            @Override
            public void run() {
                isDraining = false;
            }
        });
    }

    private void drainMessages(final MQService.MessagesDispatcher dispatcher,
                               final long fromSeqId,
                               final boolean isFirstDrain,
                               final Runnable completion) {
        Log.i(mode.sourceName(), Trace.once().method("drainMessages")
                .info("fromSeqId", seqId)
                .info("isFirstDrain", isFirstDrain));

        HMR.getService(Channel.class).run(mode.createPullingRequest(fromSeqId, isFirstDrain,
                new RichCompletionArg<RPCPullingResponse>("Source::createPullingRequest:" + mode.sourceName())
                        .onSuccess(new OnSuccessArg<RPCPullingResponse>() {
                            @Override
                            public void onSuccess(RPCPullingResponse resp) {
                                if (resp.messages.size() > 0) {
                                    try { // 加上try/catch, 避免外部消息分派处理崩溃导致drain流程被异常中断
                                        dispatcher.dispatch(resp.messages, Source.this);
                                    } catch (final Throwable t) {
                                        Log.e(mode.sourceName(), Trace.once().method("handleDrainingSuccess")
                                                .msg("Exception while dispatching messages")
                                                .info("exception", t.getLocalizedMessage()));

                                        // 外部异常不应该影响内部逻辑的继续执行，因此，不应该return！

                                        Statistics.report(Statistics.Codes.ExceptionalDispatch,
                                                new Statistics.Fields() {{
                                            StringWriter sw = new StringWriter();
                                            t.printStackTrace(new PrintWriter(sw));

                                            errInfo = sw.toString();
                                        }});
                                    }
                                }

                                // 仅在拉取成功时才可以更新lastPullAt数据，这里的设置不会影响isInitialDrain参数，因为它是在发起drain请求
                                // 之前就已经决议的
                                lastPullAt = System.currentTimeMillis();
                                Log.i(mode.sourceName(), Trace.once()
                                        .method("handleDrainingSuccess")
                                        .msg("lastPullAt -> " + lastPullAt));

                                if (resp.maxSeqId != null) {
                                    if (resp.hasMore) {
                                        // 场景：消费了消息，且还有更多后续消息
                                        // 应对：继续拉取，根据maxSeqId触发下一次拉取
                                        Source.this.drainMessages(dispatcher, resp.maxSeqId, isFirstDrain, completion);
                                    } else {
                                        // 场景：已消费完所有消息 - EOF
                                        // 应对：停止拉取，下次拉取应从maxSeqId开始
                                        HMRContext.work.async("Source::RPCPullingResponse:EOF", completion);
                                    }

                                    // 每次有效的pull操作，都应该尽可能更新seqId
                                    Source.this.setSeqId(resp.maxSeqId, true);
                                } else { // maxSeqId == null
                                    if (!resp.hasMore) {
                                        // 场景：进行了一次无效拉取，没有消费任何消息
                                        // 应对：停止拉取，下次拉取还是从fromSeqId开始
                                        HMRContext.work.async("Source::RPCPullingResponse:noMore", completion);
                                    } else {
                                        // 该场景理论上不可能出现，但确实偶发会遇到，为了避免客户端崩溃，这里只是进行了日志记录，并尝试中断
                                        // 拉取流程。并'寄望于'下次拉取可以正常恢复
                                        Log.e(mode.sourceName(), Trace.once()
                                                .method("handleDrainingSuccess")
                                                .msg("BUGGY!! 未知场景，hasMore为True, 但maxSeqId为null，无法确定下次拉取从哪开始"));

                                        HMRContext.work.async("Source::RPCPullingResponse:BUGGY", completion);

                                        Statistics.report(Statistics.Codes.ImpossibleScene, new Statistics.Fields() {{
                                            errInfo = String.format(Locale.US,
                                                    "hasMore为True, 但maxSeqId为null，"
                                                            + "无法确定下次拉取从哪开始。"
                                                            + "fromSeqId: %d", fromSeqId);
                                        }});
                                    }
                                }
                            }
                        })
                        .onFailure(new OnFailure() {
                            @Override
                            public void onFailure(Error err) {
                                HMRContext.work.async("Source::RPCPullingResponse:failure", completion);
                            }
                        })
        ));
    }

    private void setSeqId(long seqId, boolean shouldPersistent) {
        if ((this.seqId != null) && (this.seqId == seqId)) {
            // Ignore nonsense assignment
            return;
        }

        Log.i(mode.sourceName(), Trace.once().method("setSeqId")
                .msg("%d -> %d", this.seqId, seqId));

        if (shouldPersistent) {
            this.mode.storeSeqId(seqId);
        }

        this.seqId = seqId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Source source = (Source) o;
        return mode.equals(source.mode);
    }

    @Override
    public int hashCode() {
        return mode.hashCode();
    }

    public interface Mode {
        String sourceName();

        String topicName();

        MQService.FetchStrategy getStrategy();

        long getPullingPeriod();

        void loadSeqId(MQService.FetchStrategy strategy, RichCompletionArg<Long> completion);

        void storeSeqId(long seqId);

        Channel.RPC createPullingRequest(long seqId,
                                         boolean isFirstDrain,
                                         RichCompletionArg<RPCPullingResponse> completion);

        void start(Runnable changeHandler);

        void stop();
    }

    public static class Private implements Mode {

        private static final String StorageKeySeqIDPrefix = "local_sequence_id";
        private final String topic;
        private Channel.NotificationHandler changeHandler;
        private MQService.FetchStrategy strategy;
        private long pullingPeriod;

        public Private(String topic) {
            this(topic, null);
        }

        public Private(String topic, MQService.FetchStrategy strategy) {
            this(topic, strategy, DEFAULT_PULLING_PERIOD);
        }

        public Private(String topic, MQService.FetchStrategy strategy, long pullingPeriod) {
            this.topic = (topic == null) ? "" : topic;
            this.strategy = strategy;
            this.pullingPeriod = pullingPeriod < 1L ? DEFAULT_PULLING_PERIOD : pullingPeriod;
        }

        @Override
        public MQService.FetchStrategy getStrategy() {
            return this.strategy;
        }

        @Override
        public long getPullingPeriod() {
            return this.pullingPeriod;
        }

        @Override
        public String sourceName() {
            return "PrivateSource(" + topic + ")";
        }

        @Override
        public String topicName() {
            return topic;
        }

        @Override
        public void loadSeqId(MQService.FetchStrategy strategy, final RichCompletionArg<Long> completion) {
            Long seqId = null;

            if (strategy == MQService.FetchStrategy.Continuously) {
                seqId = PrefStorage.storage().execute(new PrefStorage.Query<Long>() {
                    @Override
                    public Long run(SharedPreferences pref) {
                        if (pref == null) {
                            return null;
                        }

                        long seq = pref.getLong(Private.this.prefKey(), -1);
                        if (seq == -1) {
                            // null 才是正确的"不确定、不存在、没有"语义，-1是错误的语义表达方式
                            return null;
                        } else {
                            return seq;
                        }
                    }
                });
            } else if (strategy == MQService.FetchStrategy.ReloadHistories) {
                seqId = 0L;
            }

            Log.i(sourceName(), Trace.once().method("loadSeqId").info("seqId", seqId));

            if (seqId == null) {
               if (strategy == MQService.FetchStrategy.IgnoreBefore) {
                   fetchMaxSeqId(3, topic, completion);
               } else {
                   // 如果无法确定seqId，则应尝试从服务器进行查询
                   // 1. 本地重装App
                   // 2. 强行使用SkipFetched模式
                   fetchSeqId(3, topic, completion);
               }
            } else {
                CompletionUtils.dispatchSuccess(completion, seqId);
            }
        }

        @Override
        public void storeSeqId(final long seqId) {
            PrefStorage.storage().execute(new PrefStorage.Edit() {
                @Override
                public void run(SharedPreferences.Editor editor) {
                    editor.putLong(Private.this.prefKey(), seqId);
                }
            });

            // 客户端重装时，可以进行用户上下文恢复，因为该请求为非关键请求，所以对completion不关心
            HMR.getService(Channel.class).run(new RPCReportPrivateSeqId(topic, seqId, null));
        }

        @Override
        public Channel.RPC createPullingRequest(long seqId,
                                                boolean isFirstDrain,
                                                RichCompletionArg<RPCPullingResponse> completion) {
            return new RPCPullPrivateMessages(topic, isFirstDrain, seqId, 200, null, completion);
        }

        @Override
        public void start(final Runnable sourceChanged) {
            Log.i(sourceName(), Trace.once().method("start"));

            // 服务器在个人队列新增了消息时会下发单播通知，该通知是被动的，必须进行通知注册才能进行响应
            changeHandler = new Channel.NotificationHandler() {
                @Override
                public void onNotify(String serviceName, String functionName, final byte[] data) {
                    String wantedUri = "service_api_gateway/cim.proto.PushService.IMPushMsg";
                    String actualUri = serviceName + '/' + functionName;
                    if (!wantedUri.equals(actualUri)) {
                        return;
                    }

                    HMRContext.work.async("Source::onNotify", new Runnable() {
                        @Override
                        public void run() {
                            handleNotify(data);
                        }
                    });
                }

                @Override
                public String toString() {
                    return sourceName();
                }

                private void handleNotify(byte[] data) {
                    try {
                        final Push.IMPushMsgRequest change = Push.IMPushMsgRequest
                                .newBuilder()
                                .mergeFrom(data)
                                .build();

                        if (change.getEnvName() != null
                                && !change.getEnvName().isEmpty()) {
                            if (HMRContext.region == null) {
                                Log.w(sourceName(), Trace.once().method("onNotify")
                                        .msg("localEnv is null"));
                                return;
                            }

                            HMRContext.Region remoteRegion = HMRContext.Region.make(change.getRegion() + "/" + change
                                    .getEnvType() + "/" + change.getEnvName());
                            String localEnv = HMRContext.region.toString();

                            String remoteEnv = remoteRegion.toString();
                            if (!Objects.equals(remoteEnv, localEnv)) {
                                // 收到不属于当前接入区域的消息，直接忽略

                                Log.i(sourceName(), Trace.once().method("onNotify")
                                        .msg("Ignored notify of different env")
                                        .info("localEnv", localEnv)
                                        .info("remoteEnv", remoteEnv));

                                return;
                            }
                        }

                        if (Objects.equals(topic, change.getTopic())) {
                            Log.i(sourceName(), Trace.once().method("onPrivateSourceChanged")
                                    .info("topic", change.getTopic())
                                    .info("seqId", change.getSeqId()));

                            // Notify when source changed
                            sourceChanged.run();
                        }
                    } catch (Throwable t) {
                        Log.e(sourceName(), Trace.once("Failed parsing IMPushMsgRequest").info("Exception", t));
                    }
                }
            };
            HMR.getService(Channel.class).addNotificationHandler(changeHandler);
        }

        @Override
        public void stop() {
            Log.i(sourceName(), Trace.once().method("stop"));

            HMRContext.work.async("Source::stop", new Runnable() {
                @Override
                public void run() {
                    HMR.getService(Channel.class).removeNotificationHandler(changeHandler);
                }
            });
        }

        private void fetchSeqId(final int remainTries,
                                final String topic,
                                final RichCompletionArg<Long> completion) {
            Log.i(sourceName(), Trace.once().method("fetchSeqId"));
            if (remainTries <= 0) {
                // 超过重试次数时，将seqId设置为0，这是为了保证在弱网情况下fetchReportedSequenceId无限重试，进而
                // 使得始终无法拉取消息的问题。提高了消息拉取逻辑的整体稳定性

                CompletionUtils.dispatchFailure(completion,
                        new Error(Error.Code.OperationTimeout, "Failed fetching seqId: " + topic));

                return;
            }

            // See <RichCompletion Token> Pattern
            HMR.getService(Channel.class).run(new RPCFetchPrivateSeqId(topic,
                    new RichCompletionArg<Long>("Source::RPCFetchPrivateSeqId:" + topic)
                    .onSuccess(new OnSuccessArg<Long>() {
                        @Override
                        public void onSuccess(Long seqId) {
                            CompletionUtils.dispatchSuccess(completion, seqId);
                        }
                    })
                    .onFailure(new OnFailure() {
                        @Override
                        public void onFailure(Error error) {
                            if (error != null && error.code != Error.Code.OperationTimeout) {
                                CompletionUtils.dispatchFailure(completion, error);
                            } else {
                                // 网络超时属于可恢复错误，可以再尝试几次
                                HMRContext.work.asyncAfter("Source::repeatFetch", 3 * 1000, new Runnable() {
                                    @Override
                                    public void run() {
                                        Private.this.fetchSeqId(remainTries - 1, topic, completion);
                                    }
                                });
                            }
                        }
                    })
            ));
        }

        private void fetchMaxSeqId(final int remainTries,
                                   final String topic,
                                   final RichCompletionArg<Long> completion) {
            Log.i(sourceName(), Trace.once().method("fetchMaxSeqId"));
            if (remainTries <= 0) {
                CompletionUtils.dispatchFailure(completion,
                        new Error(Error.Code.OperationTimeout, "Failed fetching seqId: " + topic));

                return;
            }

            // See <RichCompletion Token> Pattern
            HMR.getService(Channel.class).run(new RPCFetchPrivateMaxSeqId(topic,
                    new RichCompletionArg<Long>("Source::RPCFetchPrivateMaxSeqId:" + topic)
                            .onSuccess(new OnSuccessArg<Long>() {
                                @Override
                                public void onSuccess(Long seqId) {
                                    CompletionUtils.dispatchSuccess(completion, seqId);
                                }
                            })
                            .onFailure(new OnFailure() {
                                @Override
                                public void onFailure(Error error) {
                                    if (error != null && error.code != Error.Code.OperationTimeout) {
                                        CompletionUtils.dispatchFailure(completion, error);
                                    } else {
                                        // 网络超时属于可恢复错误，可以再尝试几次
                                        HMRContext.work.asyncAfter("Source::repeatFetch", 3 * 1000, new Runnable() {
                                            @Override
                                            public void run() {
                                                Private.this.fetchSeqId(remainTries - 1, topic, completion);
                                            }
                                        });
                                    }
                                }
                            })
            ));
        }

        private String prefKey() {
            String prefKey = StorageKeySeqIDPrefix;
            if (!topic.isEmpty()) {
                prefKey = prefKey + "_" + topic;
            }

            return prefKey;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Private p = (Private) o;
            return topic.equals(p.topic);
        }

        @Override
        public int hashCode() {
            return topic.hashCode();
        }
    }

    public static class Shared implements Mode {

        private static final String PrefKeySeqID = "_group_sys_seqid";
        private final long groupId;
        private final String topic;
        private Channel.NotificationHandler sharedHandler;
        private MQService.FetchStrategy strategy;
        private long pullingPeriod;

        public Shared(long groupId, String topic) {
            this(groupId, topic, null);
        }

        public Shared(long groupId, String topic, MQService.FetchStrategy strategy) {
            this(groupId, topic, strategy, DEFAULT_PULLING_PERIOD);
        }

        public Shared(long groupId,
                      String topic,
                      MQService.FetchStrategy strategy,
                      long pullingPeriod) {
            this.groupId = groupId;
            this.topic = (topic == null) ? "" : topic;
            this.strategy = strategy;
            this.pullingPeriod = pullingPeriod < 1L ? DEFAULT_PULLING_PERIOD : pullingPeriod;
        }

        @Override
        public MQService.FetchStrategy getStrategy() {
            return this.strategy;
        }

        @Override
        public long getPullingPeriod() {
            return this.pullingPeriod;
        }

        @Override
        public String sourceName() {
            return "SharedSource(" + groupId + "," + topic + ")";
        }

        @Override
        public String topicName() {
            return topic;
        }

        @Override
        public void loadSeqId(MQService.FetchStrategy strategy, RichCompletionArg<Long> completion) {
            Long seqId = null;

            if (strategy == MQService.FetchStrategy.Continuously) {
                seqId = PrefStorage.storage().execute(new PrefStorage.Query<Long>() {
                    @Override
                    public Long run(SharedPreferences pref) {
                        if (pref == null) {
                            return null;
                        }

                        long seq = pref.getLong(Shared.this.prefKey(), -1);
                        if (seq == -1) {
                            // null 才是正确的"不确定、不存在、没有"语义，-1是错误的语义表达方式
                            return null;
                        } else {
                            return seq;
                        }
                    }
                });
            } else if (strategy == MQService.FetchStrategy.ReloadHistories) {
                seqId = 0L;
            }

            Log.i(sourceName(), Trace.once().method("loadSeqId").info("seqId", seqId));

            if (seqId == null) {
                if (strategy == MQService.FetchStrategy.IgnoreBefore) {
                    fetchMaxSeqId(3, topic, completion);
                } else {
                    // 如果无法确定seqId，则应尝试从服务器进行查询
                    // 1. 本地重装App
                    // 2. 强行使用SkipFetched模式
                    fetchSeqId(3, topic, completion);
                }
            } else {
                CompletionUtils.dispatchSuccess(completion, seqId);
            }
        }

        private void fetchSeqId(final int remainTries,
                                final String topic,
                                final RichCompletionArg<Long> completion) {
            Log.i(sourceName(), Trace.once().method("Shared fetchSeqId"));
            if (remainTries <= 0) {
                // 超过重试次数时，将seqId设置为0，这是为了保证在弱网情况下fetchReportedSequenceId无限重试，进而
                // 使得始终无法拉取消息的问题。提高了消息拉取逻辑的整体稳定性

                CompletionUtils.dispatchFailure(completion,
                        new Error(Error.Code.OperationTimeout, "Shared Failed fetching seqId: " + topic));

                return;
            }

            HMR.getService(Channel.class).run(new RPCFetchSharedSeqId(topic,
                    groupId,
                    new RichCompletionArg<Long>("Source::RPCFetchSharedSeqId" + topic)
                            .onSuccess(new OnSuccessArg<Long>() {
                                @Override
                                public void onSuccess(Long seqId) {
                                    CompletionUtils.dispatchSuccess(completion, seqId);
                                }
                            })
                            .onFailure(new OnFailure() {
                                @Override
                                public void onFailure(Error error) {
                                    if (error != null && error.code != Error.Code.OperationTimeout) {
                                        CompletionUtils.dispatchFailure(completion, error);
                                    } else {
                                        // 网络超时属于可恢复错误，可以再尝试几次
                                        HMRContext.work.asyncAfter("Source::repeatFetchSeqId", 3 * 1000,
                                                new Runnable() {
                                                    @Override
                                                    public void run() {
                                                        Shared.this.fetchSeqId(remainTries - 1,
                                                                topic, completion);
                                                    }
                                                });
                                    }
                                }
                            })));
        }

        private void fetchMaxSeqId(final int remainTries,
                                final String topic,
                                final RichCompletionArg<Long> completion) {
            Log.i(sourceName(), Trace.once().method("Shared fetchMaxSeqId"));
            if (remainTries <= 0) {
                // 超过重试次数时，将seqId设置为0，这是为了保证在弱网情况下fetchReportedSequenceId无限重试，进而
                // 使得始终无法拉取消息的问题。提高了消息拉取逻辑的整体稳定性

                CompletionUtils.dispatchFailure(completion,
                        new Error(Error.Code.OperationTimeout, "Shared Failed fetching maxSeqId: " + topic));

                return;
            }

            HMR.getService(Channel.class).run(new RPCFetchSharedMaxSeqId(topic,
                    groupId,
                    new RichCompletionArg<Long>("Source::RPCFetchSharedMaxSeqId" + topic)
                            .onSuccess(new OnSuccessArg<Long>() {
                                @Override
                                public void onSuccess(Long seqId) {
                                    CompletionUtils.dispatchSuccess(completion, seqId);
                                }
                            })
                            .onFailure(new OnFailure() {
                                @Override
                                public void onFailure(Error error) {
                                    if (error != null && error.code != Error.Code.OperationTimeout) {
                                        CompletionUtils.dispatchFailure(completion, error);
                                    } else {
                                        // 网络超时属于可恢复错误，可以再尝试几次
                                        HMRContext.work.asyncAfter("Source::repeatFetchMaxSeqId", 3 * 1000,
                                                new Runnable() {
                                                    @Override
                                                    public void run() {
                                                        Shared.this.fetchSeqId(remainTries - 1,
                                                                topic, completion);
                                                    }
                                                });
                                    }
                                }
                            })));
        }

        @Override
        public void storeSeqId(final long seqId) {
            PrefStorage.storage().execute(new PrefStorage.Edit() {
                @Override
                public void run(SharedPreferences.Editor editor) {
                    editor.putLong(Shared.this.prefKey(), seqId);
                }
            });

            // 客户端重装时，可以进行用户上下文恢复，因为该请求为非关键请求，所以对completion不关心
            HMR.getService(Channel.class).run(new RPCReportSharedSeqId(topic, groupId, seqId, null));
        }

        @Override
        public Channel.RPC createPullingRequest(long seqId,
                                                boolean isFirstDrain,
                                                RichCompletionArg<RPCPullingResponse> completion) {
            return new RPCPullSharedMessages(groupId, topic, seqId, 200, isFirstDrain, completion);
        }

        @Override
        public void start(final Runnable sourceChanged) {
            Log.i(sourceName(), Trace.once().method("start"));

            sharedHandler = new Channel.NotificationHandler() {
                @Override
                public void onNotify(String serviceName, String functionName, final byte[] data) {
                    String wantedUri = "service_api_gateway/cim.proto.PushService.IMPushGroupSysMsg";
                    String actualUri = serviceName + '/' + functionName;
                    if (!wantedUri.equals(actualUri)) {
                        return;
                    }
                    HMRContext.work.async("SharedSource::onNotify", new Runnable() {
                        @Override
                        public void run() {
                            handleNotify(data);
                        }
                    });
                }

                @Override
                public String toString() {
                    return sourceName();
                }

                private void handleNotify(byte[] data) {
                    final Push.IMPushGroupSysMsgRequest request;
                    try {
                        request = Push.IMPushGroupSysMsgRequest
                                .newBuilder()
                                .mergeFrom(data)
                                .build();
                    } catch (Throwable t) {
                        Log.e(sourceName(), Trace.once("Failed parsing IMPushGroupSysMsgRequest").info("Exception", t));
                        return;
                    }

                    if (request.getEnvName() != null && !request.getEnvName().isEmpty()) {
                        if (HMRContext.region == null) {
                            Log.w(sourceName(), Trace.once().method("onNotify")
                                    .msg("localEnv is null"));
                            return;
                        }

                        HMRContext.Region remoteRegion = HMRContext.Region.make(request.getRegion() + "/" + request
                                .getEnvType() + "/" + request.getEnvName());
                        String localEnv = HMRContext.region.toString();

                        String remoteEnv = remoteRegion.toString();

                        if (!Objects.equals(remoteEnv, localEnv)) {
                            // 收到不属于当前接入区域的消息，直接忽略
                            Log.i(sourceName(), Trace.once().method("onNotify")
                                    .msg("Ignored notify of different env")
                                    .info("localEnv", localEnv)
                                    .info("notifyEnv", remoteEnv));
                            return;
                        }
                    }

                    if (Objects.equals(topic, request.getTopic())
                            && Objects.equals(groupId, request.getGroupId())) {

                        Log.i(sourceName(), Trace.once().method("onSharedSourceChanged")
                                .info("topic", request.getTopic())
                                .info("groupId", request.getGroupId())
                                .info("seqId", request.getSeqId()));

                        // Notify when source changed
                        sourceChanged.run();
                    }
                }
            };

            HMR.getService(Channel.class).addNotificationHandler(sharedHandler);
            HMR.getService(Channel.class).subscribeGroupcast(getGroup(), null);
        }

        @Override
        public void stop() {
            Log.i(sourceName(), Trace.once().method("stop"));

            HMRContext.work.async("SharedSource::stop", new Runnable() {
                @Override
                public void run() {
                    HMR.getService(Channel.class).removeNotificationHandler(sharedHandler);
                    HMR.getService(Channel.class).unSubscribeGroupcast(getGroup(), null);
                }
            });
        }

        private String getGroup() {
            return String.format(Locale.US, "hummer:%d:%s:%d", HMRContext.getAppId(), topic, groupId);
        }

        private String prefKey() {
            if (topic.isEmpty()) {
                return groupId + PrefKeySeqID;
            }
            return groupId + "_" + topic + PrefKeySeqID;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Shared shared = (Shared) o;
            return groupId == shared.groupId && topic.equals(shared.topic);
        }

        @Override
        public int hashCode() {
            return topic.hashCode() ^ (int) groupId;
        }
    }
}

