package com.hummer.im._internals.mq;

import android.support.annotation.NonNull;

import com.hummer.im.HMR;
import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.Objects;
import com.hummer.im._internals.PrefStorage;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.proto.Im;
import com.hummer.im._internals.services.mq.Statistics;
import com.hummer.im._internals.services.user.UserService;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.service.Channel;
import com.hummer.im.service.MQService;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;

public final class MQServiceImpl implements MQService,
        MQService.MessagesDispatcher,
        ServiceProvider.Service {

    private static final String TAG = "MQService";
    // Stable properties
    private final Set<Source> sources = new HashSet<>();
    // Unstable properties
    private String runningToken;
    private boolean isConnected;
    private Channel.StateChangedListener channelStateListener;
    private FetchStrategy mFetchStrategy = FetchStrategy.Continuously;

    @Override
    public Class[] staticDependencies() {
        return new Class[]{Channel.class};
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return null;
    }

    @Override
    public void initService() {
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{UserService.class, PrefStorage.class};
    }

    @Override
    public void openService(@NonNull RichCompletion completion) {
        if (runningToken != null) {
            Log.w(TAG, Trace.once().method("openService").msg("在服务已运行状态下调用了openService方法"));
            return;
        }

        runningToken = UUID.randomUUID().toString();
        if (HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchSuccess(completion);
            return;
        }

        prepareChannel();

        for (Source s : sources) {
            s.start(MQServiceImpl.this);
        }

        scheduleTimerPulse();

        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void closeService() {
        if (runningToken == null) {
            Log.w(TAG, Trace.once().method("openService").msg("在服务已关闭状态下调用了closeService方法"));
            return;
        }

        if (HMR.getMe().isAnonymous()) {
            runningToken = null;
            return;
        }

        for (Source s : sources) {
            s.stop();
        }

        teardownChannel();
        runningToken = null;
    }

    @Override
    public FetchStrategy getFetchStrategy() {
        return mFetchStrategy;
    }

    @Override
    public void setFetchStrategy(FetchStrategy strategy) {
        mFetchStrategy = strategy;
    }

    @Override
    public void addSource(final Source source) {
        HMRContext.work.async("MQServiceImpl::addSource", new Runnable() {
            @Override
            public void run() {
                Log.i(TAG, Trace.once().method("addSource")
                        .info("source", source));

                if (sources.contains(source)) {
                    Log.i(TAG, Trace.once().method("addSource")
                            .info("Duplicate source", source));
                    return;
                }

                sources.add(source);

                // Draining标记未重置的上报频率应该和sources的数量是相关的，否则同一个问题就会被上报多次
                Statistics.Frequencies.put(Statistics.Codes.NotResetIsDraining, sources.size());

                // 应立即启动CIM服务运行时过程添加的源
                if (MQServiceImpl.this.isRunning()) {
                    source.start(MQServiceImpl.this);
                }
            }
        });
    }

    @Override
    public void removeSource(final Source source) {
        HMRContext.work.async("MQServiceImpl::removeSource", new Runnable() {
            @Override
            public void run() {
                Log.i(TAG, Trace.once().method("removeSource")
                        .info("source", source));

                Source operateSource = getSource(source);

                // 应立即关闭CIM服务运行时过程添加的源，否则该源的一些相关资源可能无法正确释放
                if (MQServiceImpl.this.isRunning()) {
                    operateSource.stop();
                }

                sources.remove(operateSource);
            }
        });
    }

    @Override
    public void pullManually() {
        HMRContext.work.async("MQServiceImpl::pullManually", new Runnable() {
            @Override
            public void run() {
                Log.i(TAG, Trace.once().method("pullManually"));

                MQServiceImpl.this.notifyPullManually();
            }
        });
    }

    @Override
    public void dispatch(final List<Im.Msg> msgs, com.hummer.im._internals.mq.Source fromSource) {
        for (Im.Msg msg : msgs) {
            parseMsg(msg, fromSource);
        }
    }

    private Source getSource(Source source) {
        for (Source s : sources) {
            if (s.equals(source)) {
                return s;
            }
        }

        return source;
    }

    private void prepareChannel() {
        if (channelStateListener != null) {
            Log.e("MQServiceImpl", Trace.once().method("prepareChannel")
                    .info("channelStateListener exception", null));
            return;
        }

        isConnected = true;

        channelStateListener = new Channel.StateChangedListener() {
            @Override
            public void onChannelConnected() {
                HMRContext.work.async("MQServiceImpl::onChannelConnected", new Runnable() {
                    @Override
                    public void run() {
                        Log.w(TAG, Trace.once().method("onChannelConnected"));
                        isConnected = true;
                        notifyNetworkReconnected();
                    }
                });
            }

            @Override
            public void onChannelDisconnected() {
                HMRContext.work.async("MQServiceImpl::onChannelDisconnected", new Runnable() {
                    @Override
                    public void run() {
                        Log.w(TAG, Trace.once().method("onChannelDisconnected"));
                        isConnected = false;
                    }
                });
            }
        };
        HMR.getService(Channel.class).addStateListener(channelStateListener);
    }

    private void teardownChannel() {
        HMR.getService(Channel.class).removeStateListener(channelStateListener);
        channelStateListener = null;

        isConnected = false;
    }

    private void scheduleTimerPulse() {
        final String token = runningToken;

        final int periodMillis = 60 * 1000; // 激发时间脉冲事件的间隔，单位为毫秒
        final Runnable inspector = new Runnable() {
            @Override
            public void run() {
                // 对于长延时任务，必须进行token校验，否则可能引发非当前用户上下文的操作
                if (!Objects.equals(token, runningToken)) {
                    // Timer的终止条件是调用了closeService方法，进而清除/改变了runningToken
                    // 而一旦timer的工作上下文发生了变化，则应停止timer，直到用户重新调用了openService，会再次在另一个
                    // 独立的timer配合其上下文进行工作。
                    return;
                }

                if (!isConnected) {
                    // 通道断连时，不应该发起周期请求，例如延时消息拉取。否则会导致大量无效请求
                    return;
                }

                for (Source s : sources) {
                    s.onTimerPulse(MQServiceImpl.this);
                }

                HMRContext.work.asyncAfter("MQServiceImpl::timer:inspector", periodMillis, this);
            }
        };

        HMRContext.work.asyncAfter("MQServiceImpl::scheduleTimerPulse", periodMillis, inspector);
    }

    private boolean isRunning() {
        return runningToken != null;
    }

    private void notifyNetworkReconnected() {
        for (Source s : sources) {
            s.onNetworkReconnected(this);
        }
    }

    private void notifyPullManually() {
        for (Source s : sources) {
            s.onManualPullingRequest(this);
        }
    }

    @Override
    public void registerMsgParser(MsgParser parser) {
        mMsgParsers.add(parser);
    }

    private void parseMsg(Im.Msg msg, com.hummer.im._internals.mq.Source fromSoure) {
        for (MsgParser parser : mMsgParsers) {
            try {
                parser.parse(msg, fromSoure);
            } catch (Exception e) {
                Log.e(TAG, Trace.once().msg("parseMsg")
                        .msg("Parsing failed")
                        .info("msg", msg)
                        .info("exception", e));
            }
        }
    }

    private final List<MsgParser> mMsgParsers = new ArrayList<>();
}

