package com.hummer.im._internals.log.trace;

import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.hummer.im._internals.shared.LogId;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;

@SuppressWarnings("unused")
public final class Trace {

    public static class Flow {
        public final long logId;

        public Flow() {
            this.logId = LogId.generate();
            this.name = String.format(Locale.US, "%-19d", logId);
        }

        public Trace trace() {
            return new Trace(StackTraceInfo.getInvokingMethodName())
                    .asFlow(name, getIntervalByAdvancingTimestamp());
        }

        public Trace trace(String fmt, Object... args) {
            return new Trace(StackTraceInfo.getInvokingMethodName())
                    .asFlow(name, getIntervalByAdvancingTimestamp())
                    .msg(fmt, args);
        }

        public Trace trace(Object msgObj) {
            return new Trace(StackTraceInfo.getInvokingMethodName())
                    .asFlow(name, getIntervalByAdvancingTimestamp())
                    .msg(msgObj == null ? "null" : msgObj.toString());
        }

        private Long getIntervalByAdvancingTimestamp() {
            Long now = System.currentTimeMillis();
            Long interval = null;

            if (timestamps.size() >= 1) {
                Long last = timestamps.get(timestamps.size() - 1);
                interval = now - last;
            }

            timestamps.add(now);
            return interval;
        }

        private final String name;
        private final ArrayList<Long> timestamps = new ArrayList<>();
    }

    public static Trace once() {
        return new Trace(StackTraceInfo.getInvokingMethodName());
    }

    public static Trace once(String fmt, Object... args) {
        return new Trace(StackTraceInfo.getInvokingMethodName()).msg(fmt, args);
    }

    public static Trace once(Object msg) {
        return new Trace(StackTraceInfo.getInvokingMethodName()).msg(msg == null ? "null" : msg.toString());
    }

    public Trace msg(String fmt, Object... args) {
        this.message = String.format(Locale.US, fmt, args);
        return this;
    }

    public Trace msg(Object obj) {
        this.message = (obj == null ? "null" : obj.toString());
        return this;
    }

    public Trace method(@NonNull String name) {
        this.methodName = name;
        return this;
    }

    public Trace info(String name, Object value) {
        parameters.add(name + ":" + value);
        return this;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();

        if (flowName != null) {
            sb.append(String.format(Locale.US, "[%s]", flowName));
        }

        sb.append(methodName);

        // 过小的时耗没必要进行展示，用户通常只对那些长耗时的操作敏感
        if (flowElapsed != null && flowElapsed >= 10) {
            sb.append("(");
            sb.append(flowElapsed);
            sb.append("ms)");
        }

        if (message != null || parameters.size() > 0) {
            sb.append(" |");
        }

        if (message != null) {
            sb.append(" ");
            sb.append(message);
        }

        if (parameters.size() > 0) {
            sb.append(" {");
            sb.append(TextUtils.join(", ", parameters));
            sb.append("}");
        }

        parameters.clear();
        message = null;

        return sb.toString();
    }

    private Trace(String methodName) {
        this.methodName = methodName;
    }

    // Log.i(TAG, "[" + logId + "] function: Succeed - seqId(" + seqId + "), group(" + groupId + ")")
    // Log.i(TAG, Trace.once("Succeed").info("seqId", seqId).info("groupId", groupId))

    private Trace asFlow(String name, Long elapsed) {
        this.flowName = name;
        this.flowElapsed = elapsed;
        return this;
    }

    // [logId]function | message - info
    private final List<String> parameters = new LinkedList<>();
    private String methodName;
    private String message;

    private Long flowElapsed = null;
    private String flowName = null;
}
