package com.hummer.im._internals.chatsvc;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.google.protobuf.ByteString;
import com.hummer.im.Error;
import com.hummer.im._internals.HummerException;
import com.hummer.im._internals.IMRPC;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.proto.Chat;
import com.hummer.im._internals.proto.Im;
import com.hummer.im._internals.shared.StringChain;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.id.Group;
import com.hummer.im.model.id.User;

public class RPCSendMessage extends
        IMRPC<Chat.P2PChatRequest, Chat.P2PChatRequest.Builder, Chat.P2PChatResponse> {

    public RPCSendMessage(Message message, RichCompletion completion) {
        this.message    = message;
        this.completion = completion;
    }

    @Override
    public String getHummerFunction() {
        if (message.getReceiver() instanceof User) {
            return "P2PChat";
        } else if (message.getReceiver() instanceof Group) {
            return "GroupChat";
        }

        Log.e("RPCSendMessage", Trace.once().method("performStarting")
                    .info("不支持的消息Receiver类型: ", message.getReceiver()));
        return null;
    }

    @Override
    public void buildHummerRequest(@NonNull Chat.P2PChatRequest.Builder builder) throws Throwable {
        if (message.getPushContent() != null) {
            Im.OsPushMsg osPushContent = Im.OsPushMsg.newBuilder()
                    .setTitle(message.getPushContent().getTitle())
                    .setContent(message.getPushContent().getContent())
                    .setPayload(ByteString.copyFrom(message.getPushContent().getPayload()))
                    .setIcon(message.getPushContent().getIconUrl())
                    .build();

            builder.setOsPushMsg(osPushContent);
        }

        String extension = message.getAppExtra();
        if (extension == null) {
            extension = "";
        }

        builder.setFromUid(message.getSender().getId())
                .setToUid(message.getReceiver().getId())
                .setUuid(message.getUuid())
                .setExtension(extension);

        byte[] contentBytes = Content.makeBytes(message.getContent());
        Integer messageType = Content.getDataType(message.getContent());
        if (contentBytes == null || messageType == null) {
            throw new HummerException(Error.Code.ClientExceptions, "Encode failed");
        }

        builder.setMsgType(messageType).setContent(ByteString.copyFrom(contentBytes));
    }

    @Override
    public String describeHummerRequest(Chat.P2PChatRequest req) {
        return new StringChain().acceptNullElements()
                .add("message", message)
                .toString();
    }

    @Override
    public void handleHummerSuccess(@NonNull Chat.P2PChatResponse res) {
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void handleHummerError(@Nullable Chat.P2PChatResponse res, @NonNull Error error) {
        CompletionUtils.dispatchFailure(completion, error);
    }

    @Override
    public String describeHummerResponse(@NonNull Chat.P2PChatResponse res) {
        return null;
    }

    private final Message message;
    private final RichCompletion completion;
}
