package com.hummer.im._internals.chatsvc;

import com.google.protobuf.ByteString;
import com.hummer.im.HMR;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.proto.Chat;
import com.hummer.im._internals.proto.Im;
import com.hummer.im._internals.proto.Notify;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.PushContent;
import com.hummer.im.model.chat.contents.Notification;
import com.hummer.im.model.chat.states.Archived;
import com.hummer.im.model.chat.states.Revoked;
import com.hummer.im.model.id.Group;
import com.hummer.im.model.id.Identifiable;
import com.hummer.im.model.id.User;

public class MessageParser {

    public static final String TAG = "ChatService";

    public static Message parseContent(ByteString content, long timestamp, Im.Action action) {
        try {
            if (action == Im.Action.kP2PChat) {
                return parseP2PMessage(content, timestamp);
            } else if (action == Im.Action.kFanoutAppSessionChat) {
                return parseFanoutMessage(content, timestamp);
            } else if (action == Im.Action.kRevokeAppSessionChat) {
                return parseRevokedMessage(content, timestamp);
            } else if (action == Im.Action.kMultiCastAppSessionChat) {
                return parseMultiCastMessage(content, timestamp);
            } else if (action == Im.Action.kChatMessageModified) {
                return parseMessageModified(content, timestamp);
            } else if (action == Im.Action.kBatchP2PChat) {
                return parseBatchP2PMessage(content, timestamp);
            } else if (action == Im.Action.kMultiCastP2PChat) {
                return parseMultiCastP2PMessage(content, timestamp);
            } else if (action == Im.Action.kModifyAppSessionChat) {
                return parseModifyAppSessionMessage(content, timestamp);
            } else if (action == Im.Action.kBatchAppSessionChat) {
                return parseBatchAppSessionMessage(content, timestamp);
            } else if (action == Im.Action.kBroadCastAppSessionChat) {
                return parseBroadCastAppSessionMessage(content, timestamp);
            } else if (action == Im.Action.kBatchNotify) {
                return parseBatchNotification(content, timestamp);
            } else if (action == Im.Action.kGroupNotify) {
                return parseGroupNotification(content, timestamp);
            } else if (action == Im.Action.kInsertChatHistory) {
                return parseInsertChatHistory(content, timestamp);
            }
        } catch (Throwable t) {
            Log.e(TAG, Trace.once().method("parse")
                    .info("action", action)
                    .info("msg", content)
                    .info("exception", t.getMessage()));
        }
        return null;
    }

    private static Message parseP2PMessage(ByteString data, long timestamp) throws Throwable {
        Chat.P2PChatRequest req = Chat.P2PChatRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Content content = Content.makeContent(req.getMsgType(), req.getContent().toByteArray());
        if (content == null) {
            Log.e(TAG, Trace.once().method("parseP2PMessage").msg("Unrecognized content")
                    .info("msgType", req.getMsgType())
                    .info("content", req.getContent()));
            return null;
        }

        Message message = new Message();

        Identifiable receiver = new User(req.getToUid());

        String extension = req.getExtension();
        if ("".equals(extension)) {
            extension = null;
        }

        message.setContent(content);
        message.setSender(new User(req.getFromUid()));
        message.setReceiver(receiver);
        message.setPushContent(parsePushContent(req.getOsPushMsg()));
        message.setUuid(req.getUuid());
        message.setTimestamp(timestamp);
        message.setState(new Archived());
        message.setAppExtra(extension);

        return message;
    }

    private static Message parseFanoutMessage(ByteString data, long timestamp) throws Throwable {
        Chat.FanoutAppSessionChatRequest req = Chat.FanoutAppSessionChatRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Content content = Content.makeContent(req.getMsgType(), req.getContent().toByteArray());
        if (content == null) {
            Log.e(TAG, Trace.once().method("parseFanoutMessage").msg("Unrecognized content")
                    .info("msgType", req.getMsgType())
                    .info("content", req.getContent()));

            return null;
        }

        Message message = new Message();

        message.setContent(content);
        Identifiable sender = IdentifiableHelper.makeIdentifiable(req.getFromIdType(), req.getFromId());
        Identifiable receiver = IdentifiableHelper.makeIdentifiable(req.getToIdType(), req.getToId());
        if (sender == null || receiver == null) {
            Log.w(TAG, Trace.once().method("parseFanoutMessage").msg("Unrecognized sender/receiver")
                    .info("sender", req.getFromIdType())
                    .info("receiver", req.getToIdType()));
        }

        message.setSender(sender);
        message.setReceiver(receiver);
        message.setPushContent(parsePushContent(req.hasOsPushMsg() ? req.getOsPushMsg() : null));
        message.setUuid(req.getUuid());
        message.setTimestamp(timestamp);
        message.setState(new Archived());
        return message;
    }

    private static Message parseRevokedMessage(ByteString data, long timestamp) throws Throwable {
        Chat.RevokeAppSessionChatRequest req = Chat.RevokeAppSessionChatRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Identifiable sender = IdentifiableHelper.makeIdentifiable(req.getFromIdType(), req.getFromId());
        Identifiable receiver = IdentifiableHelper.makeIdentifiable(req.getToIdType(), req.getToId());
        if (receiver == null) {
            Log.w(TAG, Trace.once().method("parseRevokedMessage").msg("Unrecognized receiver")
                    .info("receiver", req.getToIdType()));

            return null;
        }

        Message chatMessage = new Message();
        chatMessage.setAppExtra(req.getExtra());
        chatMessage.setSender(sender);
        chatMessage.setReceiver(receiver);
        chatMessage.setTimestamp(req.getTimestamp());
        chatMessage.setUuid(req.getUuid());
        chatMessage.setState(new Revoked());

        return chatMessage;
    }

    private static Message parseMultiCastMessage(ByteString data, long timestamp) throws  Throwable {
        Chat.MultiCastAppSessionChatRequest req = Chat.MultiCastAppSessionChatRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Content content = Content.makeContent(req.getMsgType(), req.getContent().toByteArray());
        if (content == null) {
            Log.e(TAG, Trace.once().method("parseMultiCastMessage").msg("Unrecognized content")
                    .info("msgType", req.getMsgType())
                    .info("content", req.getContent()));

            return null;
        }

        Identifiable sender = IdentifiableHelper.makeIdentifiable(req.getFromIdType(), req.getFromId());
        Identifiable receiver = IdentifiableHelper.makeIdentifiable(req.getToIdType(), req.getToId());
        if (sender == null || receiver == null) {
            Log.w(TAG, Trace.once().method("parseMultiCasMessage").msg("Unrecognized sender/receiver")
                    .info("sender", req.getFromIdType())
                    .info("receiver", req.getToIdType()));
            return null;
        }

        Message chatMessage = new Message();
        chatMessage.setContent(content);
        chatMessage.setSender(sender);
        chatMessage.setReceiver(receiver);
        chatMessage.setTimestamp(timestamp);
        chatMessage.setPushContent(parsePushContent(req.hasOsPushMsg() ? req.getOsPushMsg() : null));
        chatMessage.setUuid(req.getUuid());
        chatMessage.setState(new Archived());

        return chatMessage;
    }

    private static Message parseMessageModified(ByteString data, long timestamp) throws Throwable {
        Im.ChatMessageModifiedRequest req = Im.ChatMessageModifiedRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Content content = Content.makeContent(req.getMsgType(), req.getContent().toByteArray());
        if (content == null) {
            Log.e(TAG, Trace.once().method("parseMessageModified").msg("Unrecognized content")
                    .info("msgType", req.getMsgType())
                    .info("content", req.getContent()));

            return null;
        }

        Message message = new Message();

        message.setContent(content);
        Identifiable sender = IdentifiableHelper.makeIdentifiable(req.getFromIdType(), req.getFromId());
        Identifiable receiver = IdentifiableHelper.makeIdentifiable(req.getToIdType(), req.getToId());
        if (sender == null || receiver == null) {
            Log.w(TAG, Trace.once().method("parseMessageModified").msg("Unrecognized sender/receiver")
                    .info("sender", req.getFromIdType())
                    .info("receiver", req.getToIdType()));
        }

        message.setSender(sender);
        message.setReceiver(receiver);
        message.setPushContent(null);
        message.setUuid(req.getUuid());
        message.setTimestamp(timestamp);
        message.setState(new Archived());
        return message;
    }

    private static Message parseBatchP2PMessage(ByteString data, long timestamp) throws Throwable {
        Chat.BatchP2PChatRequest req = Chat.BatchP2PChatRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Content content = Content.makeContent(req.getMsgType(), req.getContent().toByteArray());
        if (content == null) {
            Log.e(TAG, Trace.once().method("parseBatchP2PMessage").msg("Unrecognized content")
                    .info("msgType", req.getMsgType())
                    .info("content", req.getContent()));

            return null;
        }

        Message message = new Message();
        message.setContent(content);
        message.setSender(new User(req.getFromUid()));
        message.setReceiver(HMR.getMe());
        message.setPushContent(parsePushContent(req.getOsPushMsg()));
        message.setUuid(req.getUuid());
        message.setTimestamp(timestamp);
        message.setState(new Archived());
        return message;
    }

    private static Message parseMultiCastP2PMessage(ByteString data, long timestamp) throws Throwable {
        Chat.MultiCastP2PChatRequest req = Chat.MultiCastP2PChatRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Content content = Content.makeContent(req.getMsgType(), req.getContent().toByteArray());
        if (content == null) {
            Log.e(TAG, Trace.once().method("parseMultiCastP2PMessage").msg("Unrecognized content")
                    .info("msgType", req.getMsgType())
                    .info("content", req.getContent()));

            return null;
        }

        Message message = new Message();
        message.setContent(content);
        message.setSender(new User(req.getFromUid()));
        message.setReceiver(HMR.getMe());
        message.setPushContent(parsePushContent(req.getOsPushMsg()));
        message.setUuid(req.getUuid());
        message.setTimestamp(timestamp);
        message.setState(new Archived());
        return message;
    }

    private static Message parseModifyAppSessionMessage(ByteString data, long timestamp) throws Throwable {
        Chat.ModifyAppSessionChatRequest req = Chat.ModifyAppSessionChatRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Content content = Content.makeContent(req.getMsgType(), req.getContent().toByteArray());
        if (content == null) {
            Log.e(TAG, Trace.once().method("parseModifyAppSessionChat").msg("Unrecognized content")
                    .info("msgType", req.getMsgType())
                    .info("content", req.getContent()));

            return null;
        }

        Message message = new Message();

        message.setContent(content);
        Identifiable sender = IdentifiableHelper.makeIdentifiable(req.getFromIdType(), req.getFromId());
        Identifiable receiver = IdentifiableHelper.makeIdentifiable(req.getToIdType(), req.getToId());
        if (sender == null || receiver == null) {
            Log.w(TAG, Trace.once().method("parseModifyAppSessionChat").msg("Unrecognized sender/receiver")
                    .info("sender", req.getFromIdType())
                    .info("receiver", req.getToIdType()));
        }

        message.setSender(sender);
        message.setReceiver(receiver);
        message.setPushContent(null);
        message.setUuid(req.getUuid());
        message.setTimestamp(timestamp);
        message.setState(new Archived());

        // [消息修改] - 对此类 kChatMessageModified Action 消息作特殊处理，在这里截获parser后直接通知上层
        // 返回null为了 MQService 忽略此类型消息的处理

        // 用户需求变更，接口被删除，暂时作注释处理
        // notifyReeditMessage(message);

        return null;
    }

    private static Message parseBatchAppSessionMessage(ByteString data, long timestamp) throws Throwable {
        Chat.BatchAppSessionChatRequest req = Chat.BatchAppSessionChatRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Content content = Content.makeContent(req.getMsgType(), req.getContent().toByteArray());
        if (content == null) {
            Log.e(TAG, Trace.once().method("parseBatchAppSessionMessage").msg("Unrecognized content")
                    .info("msgType", req.getMsgType())
                    .info("content", req.getContent()));

            return null;
        }

        Message message = new Message();

        message.setContent(content);
        Identifiable sender = IdentifiableHelper.makeIdentifiable(req.getFromIdType(), req.getFromId());
        Identifiable receiver = IdentifiableHelper.makeIdentifiable(req.getToIdType(), req.getToId());
        if (sender == null || receiver == null) {
            Log.w(TAG, Trace.once().method("parseBatchAppSessionMessage").msg("Unrecognized sender/receiver")
                    .info("sender", req.getFromIdType())
                    .info("receiver", req.getToIdType()));
        }

        message.setSender(sender);
        message.setReceiver(receiver);
        message.setPushContent(parsePushContent(req.getOsPushMsg()));
        message.setUuid(req.getUuid());
        message.setTimestamp(timestamp);
        message.setState(new Archived());

        return message;
    }

    private static Message parseBroadCastAppSessionMessage(ByteString data, long timestamp) throws Throwable {
        Chat.BroadCastAppSessionChatRequest req = Chat.BroadCastAppSessionChatRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Content content = Content.makeContent(req.getMsgType(), req.getContent().toByteArray());
        if (content == null) {
            Log.e(TAG, Trace.once().method("parseBroadCastAppSessionMessage").msg("Unrecognized content")
                    .info("msgType", req.getMsgType())
                    .info("content", req.getContent()));

            return null;
        }

        Message message = new Message();

        message.setContent(content);
        Identifiable sender = IdentifiableHelper.makeIdentifiable(req.getFromIdType(), req.getFromId());
        Log.i(TAG, Trace.once().method("parseBroadCastAppSessionMessage").msg("sender")
                .info("sender type", req.getFromIdType())
                .info("id", req.getFromId()));

        Identifiable receiver = IdentifiableHelper.makeIdentifiable(req.getToIdType(), req.getToId());
        Log.i(TAG, Trace.once().method("parseBroadCastAppSessionMessage").msg("receiver")
                .info("receiver type", req.getToIdType())
                .info("id", req.getToId()));

        if (sender == null || receiver == null) {
            Log.w(TAG, Trace.once().method("parseBroadCastAppSessionMessage").msg("Unrecognized sender/receiver")
                    .info("sender", req.getFromIdType())
                    .info("receiver", req.getToIdType()));
        }

        message.setSender(sender);
        message.setReceiver(receiver);
        message.setPushContent(null);
        message.setUuid(req.getUuid());
        message.setTimestamp(timestamp);
        message.setState(new Archived());
        return message;
    }

    private static Message parseGroupNotification(ByteString data, long timestamp) throws Throwable {
        Notify.GroupNotifyRequest req = Notify.GroupNotifyRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Message message = new Message();
        Notification notification = new Notification(req.getChatContent() == null
                ? null : req.getChatContent().toByteArray());

        message.setContent(notification);
        message.setSender(new User(req.getFromUid()));
        message.setReceiver(new Group(req.getGroupId()));
        message.setPushContent(null);
        message.setUuid(req.getUuid());
        message.setTimestamp(timestamp);
        message.setState(new Archived());

        return message;
    }

    private static Message parseBatchNotification(ByteString data, long timestamp) throws Throwable {
        Notify.BatchNotifyRequest req = Notify.BatchNotifyRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Message message = new Message();
        Notification notification = new Notification(req.getChatContent() == null
                ? null : req.getChatContent().toByteArray());

        message.setContent(notification);
        message.setSender(new User(req.getFromUid()));
        message.setReceiver(HMR.getMe());
        message.setPushContent(parsePushContent(req.hasOsPushMsg() ? req.getOsPushMsg() : null));
        message.setUuid(req.getUuid());
        message.setTimestamp(timestamp);
        message.setState(new Archived());

        return message;
    }

    private static Message parseInsertChatHistory(ByteString data, long timestamp) throws Throwable {
        Chat.InsertChatHistoryRequest req = Chat.InsertChatHistoryRequest
                .newBuilder()
                .mergeFrom(data)
                .build();

        Content content = Content.makeContent(req.getMsgType(), req.getContent().toByteArray());
        if (content == null) {
            Log.e(TAG, Trace.once().method("parseInsertChatHistory").msg("Unrecognized content")
                    .info("msgType", req.getMsgType())
                    .info("content", req.getContent()));

            return null;
        }

        Message message = new Message();

        message.setContent(content);
        Identifiable sender = IdentifiableHelper.makeIdentifiable(req.getFromIdType(), req.getFromId());
        Identifiable receiver = IdentifiableHelper.makeIdentifiable(req.getToIdType(), req.getToId());
        if (sender == null || receiver == null) {
            Log.w(TAG, Trace.once().method("parseInsertChatHistory").msg("Unrecognized sender/receiver")
                    .info("sender", req.getFromIdType())
                    .info("receiver", req.getToIdType()));
        }

        message.setSender(sender);
        message.setReceiver(receiver);
        message.setPushContent(null);
        message.setUuid(req.getUuid());
        message.setTimestamp(timestamp);
        message.setState(new Archived());

        return message;
    }

    private static PushContent parsePushContent(Im.OsPushMsg pushMsg) {
        if (pushMsg == null) {
            return null;
        }

        return new PushContent(
                pushMsg.getTitle(),
                pushMsg.getContent(),
                pushMsg.getPayload().toByteArray(),
                pushMsg.getIcon());
    }
}
