package com.hummer.im._internals;

import android.content.Context;
import android.content.SharedPreferences;
import android.support.annotation.NonNull;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.services.user.UserService;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.id.User;

import java.util.Locale;

public final class PrefStorage implements ServiceProvider.Service {

    private static final String TAG = "PrefStorage";
    private static PrefStorage sharedStorage;
    private static int NumberOfClosing = 0;
    private static int NumberOfOpening = 0;
    private SharedPreferences preferences = null;
    private User me = null;

    // Prevent user initiation
    private PrefStorage() {
    }

    public static PrefStorage storage() {
        // 不会产生多线程竞争问题，因此不需要多线程锁、Double Check Locking等模式介入
        if (sharedStorage == null) {
            sharedStorage = new PrefStorage();
        }

        return sharedStorage;
    }

    public <ResultType> ResultType execute(Query<ResultType> query) {
        if (!isValidContext()) {
            Log.w(TAG, Trace.once().method("execute").msg("Bad Context : %s ", serviceSnapshot()));
            return null;
        }

        return query.run(this.preferences);
    }

    public void execute(Edit edit) {
        if (!isValidContext()) {
            Log.w(TAG, Trace.once().method("execute").msg("Bad Context : %s ", serviceSnapshot()));
            return;
        }

        SharedPreferences.Editor editor = preferences.edit();
        edit.run(editor);
        editor.apply();
    }

    private boolean isValidContext() {
        return (this.preferences != null) && HMR.isMe(this.me);
    }

    private String serviceSnapshot() {
        return "me: " + HMR.getMe()
                + ", isRunning: " + (preferences != null)
                + ", opening: " + NumberOfOpening
                + ", closing: " + NumberOfClosing;
    }

    public interface Query<ResultType> {
        ResultType run(SharedPreferences pref);
    }
    public interface Edit {
        void run(SharedPreferences.Editor editor);
    }

    @Override
    public Class[] staticDependencies() {
        return null;
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{UserService.class};
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return new Class[0];
    }

    @Override
    public void initService() {

    }

    @Override
    public void openService(@NonNull RichCompletion completion) {
        PrefStorage.NumberOfOpening += 1;

        if (this.me != null || this.preferences != null) {
            Log.w(TAG, Trace.once().method("open").msg("Nonsense opening: " + serviceSnapshot()));
            return;
        }

        this.me = HMR.getMe();

        if (this.me == null) {
            Log.e("PrefStorage", Trace.once().method("openService")
                    .info("无效的Me对象:" + HMR.getMe() + ", " + serviceSnapshot(), null));
            CompletionUtils.dispatchFailure(completion, new Error(Error.Code.BadUser, "User not login."));
            return;
        }

        HMRContext.Region env = HMRContext.region;
        String serviceName = String.format(Locale.US, "svc_%s_cim_%s_%s_proxy", env.area, env.type, env.name);

        String prefName = String.format(Locale.US, "im_%d_%d_%s",
                HMRContext.getAppId(),
                this.me.getId(),
                serviceName);
        this.preferences = HMRContext.getAppContext().getSharedPreferences(prefName, Context.MODE_PRIVATE);

        Log.i(TAG, Trace.once().method("open").info("prefName", prefName));

        if (this.preferences == null) {
            Log.e("PrefStorage", Trace.once().method("openService")
                    .info("无法创建Preferences文件 | " + serviceSnapshot(), null));
        }
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void closeService() {
        PrefStorage.NumberOfClosing += 1;

        Log.i(TAG, Trace.once().method("close"));

        if (!isValidContext()) {
            Log.w(TAG, Trace.once().method("close").msg("Nonsense closing: ", serviceSnapshot()));
            return;
        }

        this.preferences = null;
        this.me = null;
    }
}


