package com.hummer.im;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im._internals.HMRContext;
import com.hummer.im._internals.log.HummerLogger;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.services.user.UserService;
import com.hummer.im._internals.shared.DispatchQueue;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im.model.auth.TokenProvider;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.id.Identifiable;
import com.hummer.im.model.id.User;
import com.hummer.im.service.Channel;
import com.yy.yylogger.Logger;

import java.lang.ref.WeakReference;
import java.util.HashSet;
import java.util.Set;

/**
 * Hummer类型是服务的总入口，主要包含了SDK初始化，以及用户上下文切换的功能
 * <br>实际业务接入时，必须遵循下述原则：
 * <br>1. 需要在init之前，调用Hummer.registerChannel方法注册相应的Channel对象实例
 *        如果没有注册将默认为SDK提供的自治模式（AutonomousMode）的ServiceChannel
 * <br>2. 必须在App初始化时，依赖Hummer的模块初始化之前，调用Hummer.init方法，进行初始化
 * <br>3. 在用户身份确定（成功登录）后，调用{@link HMR#open}启用IM用户上下文
 * <br>4. 在用户身份失效（退出登录）后，调用{@link HMR#close}注销IM用户上下文
 */
public final class HMR {

    /**
     * Hummer SDK的初始化方法，关于运行环境的参数，欢迎垂询
     *
     * @param appContext Android应用AppContext实例对象
     * @param appId      应用appId，它是区分不同业务的重要标识。目前需要和Hummer服务提供方通过人工申请
     * @param provider   用于服务初始化时所需token的获取
     * @since 2.6.11
     */
    @Deprecated
    public static void init(@NonNull final Context appContext,
                            final long appId,
                            @Nullable final TokenProvider provider) {
        HMRContext.work.sync("HMR::init", new Runnable() {
            @Override
            public void run() {
                if (isInitialized()) {
                    Log.e("HMR", Trace.once().method("init")
                            .info("Hummer already been initialized!", ""));
                    return;
                }

                HMRContext.appId = appId;
                HMRContext.appContext = new WeakReference<>(appContext.getApplicationContext());

                if (Log.getLogger() == null) {
                    Logger.init(appId, appContext);
                    Log.setLogger(new HummerLogger("HMR"));
                }

            /*
               - `Autonomous` --> 提供实际provider
               -   `Delegate` --> provider置null
               -    自定义通道 --> 通过ServiceProvider.registerService方法设置channel，且provider传null
             */
                if (getService(Channel.class) == null) {
                    Channel channel;
                    if (provider == null) {
                        channel = new ServiceChannel(new ServiceChannel.DelegateMode());
                    } else {
                        channel = new ServiceChannel(new ServiceChannel.AutonomousMode(
                                ServiceChannel.AutonomousMode.THIRD_USER_TOKEN,
                                provider
                        ));
                    }
                    ServiceProvider.register(Channel.class, channel);
                }
                ServiceProvider.loadServicesIfNeeded(appContext, "com.hummer.im");
                HMR.state = State.Closed;
            }
        });
    }

    /**
     * Hummer SDK的初始化方法，关于运行环境的参数，欢迎垂询
     *
     * <br> 1. 业务需要在init之前register相应的Channel对象实例
     * <br> 2. 如果没有register想默认为SDK提供的自治模式（AutonomousMode）的ServiceChannel
     *
     * @param appContext Android应用AppContext实例对象
     * @param appId      应用appId，它是区分不同业务的重要标识。目前需要和Hummer服务提供方通过人工申请
     */
    public static void init(@NonNull final Context appContext,
                            final long appId) {
        HMRContext.work.sync("HMR::init", new Runnable() {
            @Override
            public void run() {
                if (isInitialized()) {
                    Log.e("HMR", Trace.once().method("init")
                            .info("Hummer already been initialized!", ""));
                    return;
                }

                HMRContext.appId = appId;
                HMRContext.appContext = new WeakReference<>(appContext.getApplicationContext());

                if (Log.getLogger() == null) {
                    Logger.init(appId, appContext);
                    Log.setLogger(new HummerLogger("HMR"));
                }

                /* 如果没有设置，则表示是自治模式。新建一个，provider时机移到 */
                if (getService(Channel.class) == null) {
                    Channel channel = new ServiceChannel(new ServiceChannel.AutonomousMode(
                            ServiceChannel.AutonomousMode.THIRD_USER_TOKEN, null
                    ));

                    ServiceProvider.register(Channel.class, channel);
                }

                ServiceProvider.loadServicesIfNeeded(appContext, "com.hummer.im");
                HMR.state = State.Closed;
            }
        });
    }

    /**
     * 切换至指定用户的工作上下文，该方法应该在上层业务的用户登录操作完成之后执行
     *
     * @param uid        用户uid，传入0时，为匿名登录
     * @param region     启用用户上下文所需的区域配置参数
     * @param tag        用于消息过滤
     * @param completion 异步操作回调，可为null，会在操作结束时通过其不同的回调方法返回操作结果。
     * @since 2.6.11
     */
    @Deprecated
    public static void open(final long uid,
                            @NonNull final String region,
                            @Nullable final Set<String> tag,
                            @Nullable final HMR.Completion completion) {
        performServiceAvailableCheck();
        final RichCompletion richcompletion = new RichCompletion(completion, "HMR.open");

        HMRContext.work.async("HMR::open", new Runnable() {
            @Override
            public void run() {
                if (getState() != State.Closed) {
                    // open是不可重入的，当在已open的状态下，应忽略可能会导致状态不一致的open请求
                    // 导致该问题的原因通常是因为业务模块较多，且没有正确处理登录状态维护
                    Log.w(TAG, Trace.once().method("open")
                            .msg("用户上下文重入")
                            .info("currentUID", getMe())
                            .info("requestUID", uid));

                    CompletionUtils
                            .dispatchFailure(richcompletion, new Error(Error.Code.BadUser, "Replicated HMR.open"));
                    return;
                }

                Log.i(TAG, Trace.once().method("open")
                        .msg("open sdk")
                        .info("uid", uid));

                setHummerState(State.Opening);
                HMR.me = new User(uid);
                HMRContext.region = HMRContext.Region.make(region);
                HMR.tags = tag;

                ServiceProvider.openServices(richcompletion
                        .beforeFailure(new Runnable() {
                            @Override
                            public void run() {
                                HMRContext.region = null;
                                HMR.me = null;
                                setHummerState(State.Closed);
                            }
                        })
                        .beforeSuccess(new Runnable() {
                            @Override
                            public void run() {
                                setHummerState(State.Opened);
                                if (HMR.tags != null) {
                                    HMR.getService(UserService.class).setTags(HMR.tags, null);
                                }
                                Log.i(TAG, Trace.once().method("open").msg("finish"));
                            }
                        })
                );
            }
        });
    }

    /**
     * 切换至指定用户的工作上下文，该方法应该在上层业务的用户登录操作完成之后执行
     *
     * @param uid        用户uid，传入0时，为匿名登录
     * @param region     启用用户上下文所需的区域配置参数
     * @param tag        用于消息过滤
     * @param provider   用于服务初始化时所需token的获取
     * @param completion 异步操作回调，可为null，会在操作结束时通过其不同的回调方法返回操作结果。
     */
    public static void open(final long uid,
                            @NonNull final String region,
                            @Nullable final Set<String> tag,
                            @Nullable final TokenProvider provider,
                            @Nullable final HMR.Completion completion) {
        performServiceAvailableCheck();
        final RichCompletion richcompletion = new RichCompletion(completion, "HMR.open");

        HMRContext.work.async("HMR::open", new Runnable() {
            @Override
            public void run() {
                if (getState() != State.Closed) {
                    // open是不可重入的，当在已open的状态下，应忽略可能会导致状态不一致的open请求
                    // 导致该问题的原因通常是因为业务模块较多，且没有正确处理登录状态维护
                    Log.w(TAG, Trace.once().method("open")
                            .msg("用户上下文重入")
                            .info("currentUID", getMe())
                            .info("requestUID", uid));

                    CompletionUtils
                            .dispatchFailure(richcompletion, new Error(Error.Code.BadUser, "Replicated HMR.open"));
                    return;
                }

                Channel channel = getService(Channel.class);
                // 如果channel是 ServiceChannel 且是自治模式
                if (channel instanceof ServiceChannel
                        && ((ServiceChannel) channel).getMode() instanceof ServiceChannel.AutonomousMode) {
                    ServiceChannel.AutonomousMode autonomousMode
                            = (ServiceChannel.AutonomousMode) ((ServiceChannel) channel).getMode();

                    if (provider == null) {
                        if (autonomousMode.getTokenProvider() == null) {
                            Log.e(TAG, Trace.once().method("open").msg("provider can not be null"));

                            CompletionUtils.dispatchFailure(richcompletion,
                                    new Error(Error.Code.InvalidParameters, "provider can not be null"));
                            return;
                        }
                    } else {
                        autonomousMode.setTokenProvider(provider);
                    }
                }

                Log.i(TAG, Trace.once().method("open")
                        .msg("open sdk")
                        .info("uid", uid));

                setHummerState(State.Opening);
                HMR.me = new User(uid);
                HMRContext.region = HMRContext.Region.make(region);
                HMR.tags = tag;

                ServiceProvider.openServices(richcompletion
                        .beforeFailure(new Runnable() {
                            @Override
                            public void run() {
                                HMRContext.region = null;
                                HMR.me = null;
                                setHummerState(State.Closed);
                            }
                        })
                        .beforeSuccess(new Runnable() {
                            @Override
                            public void run() {
                                setHummerState(State.Opened);
                                if (HMR.tags != null) {
                                    HMR.getService(UserService.class).setTags(HMR.tags, null);
                                }
                                Log.i(TAG, Trace.once().method("open").msg("finish"));
                            }
                        })
                );
            }
        });
    }

    /**
     * Hummer包含了许多服务接口，例如ChatService, ConversationService，所有服务的实例都必须通过getService方法来获取
     *
     * @return 如果正确加载、或者确实可以提供某个Service实例，则返回对应的实现实例，否则返回null
     */
    public static <Service> Service getService(@NonNull Class<Service> serviceClass) {
        // ！！该方法是提供给业务调用的，Hummer内部应直接使用{@link ServiceProvider#get}来获取service实例
        return ServiceProvider.get(serviceClass);
    }

    /**
     * 关闭Hummer的功能，该操作应该在业务进行实际的用户注销（退出登录）之前执行
     *
     * @param completion 完成关闭的一个回调
     */
    public static void close(@Nullable final Completion completion) {
        performServiceAvailableCheck();

        final RichCompletion richcompletion = new RichCompletion(completion, "HMR.close");

        HMRContext.work.async("HMR::close", new Runnable() {
            @Override
            public void run() {
                if (getState() != State.Opened) {
                    Log.w(TAG, Trace.once().method("close").msg("Miss-match closing"));
                    CompletionUtils
                            .dispatchFailure(richcompletion, new Error(Error.Code.BadUser, "Miss-match closing"));
                    return;
                }

                Log.i(TAG, Trace.once().method("close"));

                setHummerState(State.Closing);

                ServiceProvider.closeServices();

                setHummerState(State.Closed);
                HMR.me = null;
                HMRContext.region = null;
                // FIXME: 这里需要斟酌一下，ServiceProvider这里就要改成异步
                // YYServiceSDK 这边的 unbind 方法就是个异步的行为
                CompletionUtils.dispatchSuccess(richcompletion);
                Log.i(TAG, Trace.once().method("close").msg("finish"));
            }
        });
    }

    /**
     * 注册chanel
     *
     * @param channel channel实例
     */
    public static void registerChannel(Channel channel) {
        ServiceProvider.register(Channel.class, channel);
    }

    public enum State {
        Unavailable,
        Opening,
        Opened,
        Closing,
        Closed,
    }

    public interface StateListener {
        void onUpdateHummerState(State fromState, State toState);
    }

    /**
     * 获取Hummer当前的工作状态
     */
    public static State getState() {
        return HMR.state;
    }

    public static void addStateListener(@NonNull final StateListener listener) {
        HMRContext.work.async("HMR::addStateListener", new Runnable() {
            @Override
            public void run() {
                synchronized (mListeners) {
                    mListeners.add(listener);
                    Log.d(TAG, Trace.once()
                            .method("addStateListener")
                            .info("size", mListeners.size()));
                }

                setHummerState(null);
            }
        });
    }

    public static void removeStateListener(@NonNull final StateListener listener) {
        HMRContext.work.async("HMR::removeStateListener", new Runnable() {
            @Override
            public void run() {
                synchronized (mListeners) {
                    mListeners.remove(listener);
                    Log.d(TAG, Trace.once()
                            .method("removeStateListener")
                            .info("size", mListeners.size()));
                }
            }
        });
    }

    private static void setHummerState(@Nullable final HMR.State state) {
        HMR.State fromState = HMR.state;
        if (state != null) {
            HMR.state = state;
        }

        notifyUpdateHummerState(fromState, HMR.state);
    }

    private static void notifyUpdateHummerState(final HMR.State fromState, final HMR.State toState) {
        DispatchQueue.main.sync("HMR::notifyUpdateHummerState", new Runnable() {
            @Override
            public void run() {
                synchronized (mListeners) {
                    for (final StateListener l : mListeners) {
                        l.onUpdateHummerState(fromState, toState);
                    }
                }
            }
        });
    }

    /**
     * 获取Hummer的版本号
     */
    public static String getVersion() {
        return BuildConfig.BUILD_VERSION;
    }

    /**
     * 获取当前Hummer工作上下文的用户对象
     *
     * @return 如果已经成功通过HMR.open方法启用了用户上下文，则返回对应的用户实例，否则返回null。可以通过返回值来判断Hummer是否已经
     * 正确开启了用户上下文。
     */
    public static User getMe() {
        return me;
    }

    /**
     * 获取用户标签集
     */
    public static Set<String> getTags() {
        return tags == null ? new HashSet<String>() : tags;
    }

    /**
     * 判断某个标识对象是否当前Hummer的工作用户
     *
     * @param identity 欲比对的标识对象
     * @return 如果identity对象实际User类型，且其id值与当前Hummer用户的id值一致，则返回true，否则返回false
     */
    public static boolean isMe(Identifiable identity) {
        return Identifiable.equals(me, identity);
    }

    public interface Completion {
        void onSuccess();

        void onFailed(Error err);
    }

    public interface CompletionArg<Argument> {
        void onSuccess(Argument arg);

        void onFailed(Error err);
    }


    // -- Mark: Private implementations

    private HMR() {
    }

    private static void performServiceAvailableCheck() {
        String failure = "Hummer.init has to be called before acquiring it's services.";
        if (!isInitialized()) {
            Log.e("HMR", failure);
        }
    }

    private static boolean isInitialized() {
        return HMRContext.appId != null || HMR.state != State.Unavailable;
    }

    private static final Set<StateListener> mListeners = new HashSet<>();
    private static final String TAG = "Hummer";
    private static State state = State.Unavailable;
    private static User me;

    /**
     * 设置用户标签集
     *
     * <br> 用户标签会造成下述影响：
     * <br> 1. Push的发送会根据用户标签进行匹配发送
     * <br> 2. 共享消息在拉取时，会根据标签进行过滤
     */
    private static Set<String> tags;

}
