package com.hummer.im.service;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.hummer.im.HMR;
import com.hummer.im.model.chat.AppContent;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.MessageOptions;
import com.hummer.im.model.chat.contents.Audio;
import com.hummer.im.model.chat.contents.CustomContent;
import com.hummer.im.model.chat.contents.Image;
import com.hummer.im.model.chat.contents.Text;
import com.hummer.im.model.chat.contents.Video;
import com.hummer.im.model.chat.store.MessageStoreStrategy;
import com.hummer.im.model.id.Identifiable;

/**
 * ChatService是Hummer服务的核心类型，用于聊天消息的发送、接收处理.
 */
@SuppressWarnings("unused")
public interface ChatService {

    /**
     * 消息发送的接口方法，使用该方法可以发送一条ChatMessage，无论是文本消息、图片消息、语音消息，都可以通过
     * 该方法来进行发送。具体的消息类型，是由{@link Message}的实际类型来决定的。如果业务希望发送自定义
     * 类型的消息，可以自定义ChatContent类型来实现。
     * <p>
     * 重要！completion只处理了最终完成回调，但消息发送过程中可能会有多个阶段/状态，例如图片消息需要一个较长
     * 的图片上传过程，图片完成上传后的CIM服务器RPC过程。应通过对ChatMessage对象进行状态观察来进行UI渲染。
     * <p>
     * 本地不会收到自己发出去的消息，但是如果多台设备登录了同一个uid，则非发送端是可以收到的。
     * <p>
     * {@link Message}
     *
     * @param message    待发送的消息对象
     * @param completion 发送操作的完成回调
     */
    void send(@NonNull Message message, @Nullable HMR.Completion completion);

    /**
     * 创建文本消息内容
     *
     * @param text 文本内容
     *
     * @return Text
     */
    Text createText(String text);

    /**
     * 创建图片消息内容
     *
     * @param path   图片文件路径
     * @param width  图片宽
     * @param height 图片高
     *
     * @return Image
     */
    Image createImage(@NonNull String path, int width, int height);

    /**
     * 创建音频消息内容
     *
     * @param path     音频文件路径
     * @param duration 音频时长，单位毫秒(ms)
     *
     * @return Audio
     */
    Audio createAudio(@NonNull String path, int duration);


    /**
     * 创建视频消息内容
     *
     * @param videoPath     视频文件路径
     * @param videoSize     视频大小
     * @param videoWidth    视频width
     * @param videoHeight   视频height
     * @param videoDuration 视频时长
     * @param coverPath     封面图文件路径
     * @param coverWidth    封面图width
     * @param coverHeight   封面图height
     *
     * @return Video
     */
    Video createVideo(@NonNull String videoPath,
                      long videoSize,
                      int videoWidth,
                      int videoHeight,
                      int videoDuration,
                      @NonNull String coverPath,
                      int coverWidth,
                      int coverHeight);

    /**
     * 创建透传消息
     *
     * @param type 消息类型
     * @param data 消息内容
     *
     * @return AppContent
     */
    AppContent createAppContent(int type, byte[] data);

    /**
     * 创建自定义消息
     *
     * @param data 消息内容
     *
     * @return CustomContent
     */
    CustomContent createCustomContent(byte[] data);

    /**
     * 创建消息
     *
     * @param target         消息接收目标
     * @param messageContent 消息内容
     *
     * @return Message
     */
    Message createMessage(@NonNull Identifiable target, @NonNull Content messageContent);

    /**
     * 创建消息
     *
     * @param target         消息接收目标
     * @param messageContent 消息内容
     *
     * @return Message
     */
    Message createMessage(@NonNull Identifiable target,
                          @NonNull Content messageContent,
                          @NonNull MessageStoreStrategy storeStrategy);

    /**
     * 发送消息
     *
     * @param message    消息
     * @param options    消息可选项
     * @param completion 回调
     */
    void sendMessage(@NonNull Message message, MessageOptions options, HMR.Completion completion);

    /**
     * 消息服务监听器
     *
     * @since 2.16.0
     */
    interface MessageServiceListener {
        /**
         * 新消息通知
         *
         * @param message 消息
         */
        void onMessageReceived(@NonNull Message message);

        /**
         * 消息状态变更通知
         *
         * @param message 消息
         */
        void onMessageStateUpdated(@NonNull Message message);

        /**
         * 消息内容准备的进度回调。比如图片、音频的消息在上传时，会通告该回调告知其具体的上传进度
         *
         * @param message 消息
         */
        void onContentProgressUpdated(@NonNull Message message);
    }

    /**
     * 添加消息服务的监听器
     *
     * @param listener 监听器
     */
    void addMessageServiceListener(MessageServiceListener listener);

    /**
     * 移除消息服务的监听器
     *
     * @param listener 监听器
     */
    void removeMessageServiceListener(MessageServiceListener listener);

    /**
     * ChatService.Listener接口用于进行消息接收监听.
     * <p>
     * {@link ChatService#addMessageListener(Identifiable, MessageListener)}
     * {@link ChatService#removeMessageListener(Identifiable, MessageListener)}
     *
     * @deprecated 于2.16.0版本废弃，由{@link MessageServiceListener} 替代
     */
    @Deprecated
    interface MessageListener {
        /**
         * 当用户调用{@link ChatService#send(Message, HMR.Completion)}方法，在进行消息发送前，
         * 该回调。它一般用于在发送消息前将其添加到会话渲染界面，或者是消息入库等
         *
         * @param message ChatService将要发送的消息
         *
         * @deprecated 于 2.16.0 b版本废弃，由 {@link MessageServiceListener#onMessageStateUpdated(Message)} 替代
         */
        @Deprecated
        void beforeSendingMessage(@NonNull Message message);

        /**
         * 当用户调用{@link ChatService#send(Message, HMR.Completion)}进行消息发送完成（无论成功或失败）
         * 后触发的回调。
         *
         * @param message ChatService进行发送处理的消息
         *
         * @deprecated 于 2.16.0 b版本废弃，由 {@link MessageServiceListener#onMessageStateUpdated(Message)} 替代
         */
        @Deprecated
        void afterSendingMessage(@NonNull Message message);

        /**
         * 在消息接收、处理前，会执行该回调。
         *
         * @param message 消息
         *
         * @deprecated 于 2.16.0 b版本废弃，由 {@link MessageServiceListener#onMessageReceived} 替代
         */
        @Deprecated
        void beforeReceivingMessage(@NonNull Message message);

        /**
         * 当接收到新的聊天消息时，会通告onReceiveChatMessages方法进行回调。考虑到例如程序启动场景下，可能会一次性
         * 收到大量的聊天消息，如果每次都回调一条，会导致业务的UI刷新过于频繁（通常在收到消息时会将消息添加到已有消息
         * 容器中，并reload界面），从而严重降低UI渲染效率。因此，消息接口在设计时，可以采用了批量回调的方式进行处理。
         *
         * @param message ChatService所接收到的消息对象
         *
         * @deprecated 于 2.16.0 b版本废弃，由 {@link MessageServiceListener#onMessageReceived} 替代
         */
        @Deprecated
        void afterReceivingMessage(@NonNull Message message);
    }

    /**
     * 增加一个消息接收监控回调对象.
     *
     * @param target   欲监听消息接收的消息对象，当传null时监听全量消息接收通知
     * @param listener 欲增加的listener对象，如果此前该listener已经添加，则addListener相当于空操作
     */
    void addMessageListener(@Nullable Identifiable target, @NonNull MessageListener listener);

    /**
     * 移除特定的消息接收监控回调对象
     *
     * @param target   欲监听消息接收的消息对象，当传null时监听全量消息接收通知
     * @param listener 欲移除的listener对象，如果该listener不存在，则什么都不会发生
     */
    void removeMessageListener(@Nullable Identifiable target, @NonNull MessageListener listener);

    /**
     * ChatService.StateListener 接口用来监听消息状态的回调
     * 由于消息发送过程中，有些消息需要中间处理
     *
     * @deprecated 于 2.16.0 版本废弃，由 {@link MessageServiceListener} 替代
     */
    interface StateListener {

        /**
         * 当发送消息后可以通过该接口进行监听消息的状态，当发生消息状态变化时会收到这个回调的通知
         *
         * @param message 消息对象
         * @param state   消息当前的状态
         *
         * @deprecated 于 2.16.0 版本废弃，由 {@link MessageServiceListener#onMessageStateUpdated} 替代
         */
        @Deprecated
        void onUpdateMessageState(@NonNull Message message, @NonNull Message.State state);
    }

    /**
     * 为某个特定的消息添加消息状态变化监控的回调对象
     * <p>
     * 添加listener同时，sdk会触发initial回调，把当前message的state通过onUpdateMessageState方式进行回调通知
     *
     * @param message  欲监听状态变化的消息对象，当传null时监听全量消息状态变化
     * @param listener 欲增加的listener对象，如果此前该listener已经添加，则addListener相当于空操作
     */
    void addStateListener(@Nullable Message message, @NonNull StateListener listener);

    /**
     * 移除特定的消息的状态变化监控回调对象
     *
     * @param message  欲移除状态变化的消息对象，当传null时移除全量消息状态变化的监听
     * @param listener 欲移除的listener对象，如果该listener不存在，则什么都不会发生
     */
    void removeStateListener(@Nullable Message message, @NonNull StateListener listener);

}