package com.hummer.im.model.chat.store;

import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.hummer.im.model.chat.Message;
import com.hummer.im.model.id.Identifiable;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 【本地/服务器】获取会话对应消息时的条件集
 */
public final class FetchingClauses {

    /**
     * 最大消息数量。服务器目前对该值进行了限制，取值范围为: [1, 100]。如果使用null值，则默认使用最大值(100)。
     */
    private Integer limit;

    /**
     * 消息锚点：查询beforeMessage消息之前的消息
     * <br> 优先使用显式设置的uuid和beforeTimestamp，如果为空，则使用beforeMessage的
     * <br> 本地消息建议设置beforeMessage
     */
    private Message beforeMessage;

    /**
     * 消息的uuid;
     * <br> 由于同一毫秒可能有多条消息，只根据时间戳查询会有消息遗漏，加上uuid，查询uuid之前消息
     * <br> 优先使用该显式设置的uuid
     */
    private String uuid;

    /**
     * 时间戳（毫秒）：查询beforeTime(不包含)时间之前的消息
     * <br> 优先使用该显式设置的beforeTimestamp
     * <br> 如果没有设置：远端默认为查询服务器当前时间之前的消息，本地将从最新的查询
     */
    private Long beforeTimestamp;

    /**
     * 消息所属的关系目标：不设置表示查询当前会话下所有目标消息
     */
    private Set<Identifiable> senders;

    /**
     * 消息类型：不设置表示查询所有消息类型的消息
     */
    private Set<Integer> msgTypes;

    public FetchingClauses setLimit(@NonNull Integer limit) {
        this.limit = limit;
        return this;
    }

    @Nullable
    public Integer getLimit() {
        return limit;
    }

    public FetchingClauses setBeforeMessage(@NonNull Message beforeMessage) {
        this.beforeMessage = beforeMessage;
        return this;
    }

    @Nullable
    public Message getBeforeMessage() {
        return beforeMessage;
    }

    public FetchingClauses setBeforeTimestamp(@NonNull Long beforeTimestamp) {
        this.beforeTimestamp = beforeTimestamp;
        return this;
    }

    public String getUuid() {
        return uuid;
    }

    public FetchingClauses setUuid(String uuid) {
        this.uuid = uuid;
        return this;
    }

    @Nullable
    public Long getBeforeTimestamp() {
        return beforeTimestamp;
    }

    public FetchingClauses setSenders(Set<Identifiable> senders) {
        this.senders = senders;
        return this;
    }

    @Nullable
    public Set<Identifiable> getSenders() {
        return senders;
    }

    public Set<Integer> getMsgTypes() {
        return msgTypes;
    }

    public void setMsgTypes(Set<Integer> msgTypes) {
        this.msgTypes = msgTypes;
    }

    @Override
    public String toString() {
        String string = "FetchingClauses{";

        if (limit == null && beforeMessage == null) {
            string += "All";
        } else {
            List<String> parts = new ArrayList<>();

            if (beforeMessage != null) {
                parts.add("before: " + beforeMessage);
            }

            if (limit != null) {
                parts.add("limit: " + limit);
            }

            if (beforeTimestamp != null) {
                parts.add("beforeTime: " + beforeTimestamp);
            }

            if (senders != null) {
                parts.add("senders.size: " + senders.size());
            }

            if (msgTypes != null) {
                parts.add("msgTypes: " + msgTypes);
            }

            string += TextUtils.join(", ", parts);
        }

        string += "}";

        return string;
    }
}
