package com.hummer.im.model.chat.states;

import com.hummer.im.model.chat.Message;

import java.util.Locale;

/**
 * 发送中状态，表示该消息正在尝试进行Content处理，对于普通文本消息、URL消息等轻量消息来说，这个过程几乎是瞬间完成的。
 * 但对于像图片消息、语音消息来说，需要一个可观的处理（上传）过程。StateContentProcessing用于标识，消息发送正处于
 * 这个处理阶段，具体进度值由mProgress属性来标识。
 */
public final class Preparing implements Message.State {

    /**
     * 准备过程会包含一个进度对象，Progress用于标识进度的抽象概念。目前Hummer内部仅包含ScalarProgress进度类型。但是业务也可以
     * 实现该接口来表示自身支持的进度类型。
     */
    public interface Progress { }

    /**
     * 用于标识一个数值标量型的进度，通常用于文件上传、可被标量化的处理进度
     */
    public static final class ScalarProgress implements Progress {
        /**
         * 表示进度的具体数值，通常取值区间为[0, 1]
         */
        public final float value;

        /**
         * 构造一个ScalarProgress实例
         */
        public ScalarProgress(float value) {
            this.value = value;
        }

        @Override
        public String toString() {
            return String.format(Locale.US, "%.2f", value);
        }
    }

    /**
     * 获取进度实例对象
     */
    public Progress getProgress() {
        return progress;
    }

    /**
     * 构造一个通用的表示准备中的状态
     */
    public Preparing() { }

    /**
     * 构造一个包含了具体进度的"准备中"的状态实例
     * @param progress 具体的进度实例
     */
    public Preparing(Progress progress) {
        this.progress = progress;
    }

    @Override
    public String toString() {
        if (progress == null) {
            return "[Preparing]";
        } else {
            return "[Preparing: " + progress + "]";
        }
    }

    private Progress progress = null;
}

