package com.hummer.im.model.chat.contents;

import android.media.MediaMetadataRetriever;
import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.hummer.im.Error;
import com.hummer.im.ErrorEnum;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.Trace;
import com.hummer.im._internals.utility.FileUtils;
import com.hummer.im.model.chat.Content;

import org.json.JSONObject;

import java.io.File;
import java.util.Locale;

public class Video extends Content {
    private static final String TAG = "Video";

    public static final int TYPE = 4;

    /**
     * 视频路径
     */
    private final String videoUrl;

    /**
     * 视频大小
     */
    private final long videoSize;

    /**
     * 视频时长，单位毫秒
     */
    private final int videoDuration;

    /**
     * 视频宽
     */
    private final int videoWidth;

    /**
     * 视频高
     */
    private final int videoHeight;

    /**
     * 封面图路径
     */
    private final String coverUrl;

    /**
     * 封面缩略图路径
     */
    private String coverThumbUrl;

    /**
     * 封面图宽
     */
    private final int coverWidth;

    /**
     * 封面图高
     */
    private final int coverHeight;

    private Video(String videoUrl,
                  long videoSize,
                  int videoWidth,
                  int videoHeight,
                  int videoDuration,
                  String coverUrl,
                  int coverWidth,
                  int coverHeight) {
        this.videoUrl = videoUrl;
        this.videoSize = videoSize;
        this.videoDuration = videoDuration;
        this.videoWidth = videoWidth;
        this.videoHeight = videoHeight;
        this.coverWidth = coverWidth;
        this.coverHeight = coverHeight;
        this.coverUrl = coverUrl;
    }


    public static Video create(@NonNull String videoPath,
                               long videoSize,
                               int videoWidth,
                               int videoHeight,
                               int videoDuration,
                               @NonNull String coverPath,
                               int coverWidth,
                               int coverHeight) {
        Log.i(TAG, Trace.method("create")
                .info("video", videoPath)
                .info("videoSize", videoSize)
                .info("videoDuration", videoDuration)
                .info("videoWidth", videoWidth)
                .info("videoHeight", videoHeight)
                .info("coverPath", coverPath)
                .info("coverWidth", coverWidth)
                .info("coverHeight", coverHeight));

        if (!TextUtils.isEmpty(coverPath)) {
            Image coverImage = Image.create(coverPath, coverWidth, coverHeight);
            if (coverImage != null) {
                coverWidth = coverImage.getWidth();
                coverHeight = coverImage.getHeight();
            }
        }

        if (FileUtils.isWebUrl(videoPath)) {
            Log.w(TAG, Trace.method("create").msg("Create url Video"));

            return new Video(videoPath, videoSize, videoWidth, videoHeight, videoDuration,
                    coverPath, coverWidth, coverHeight);
        }

        /* 如果视频都不存在，直接返回 */
        File videoFile = new File(videoPath);
        boolean videoFlag = videoFile.exists() && !videoFile.isDirectory();
        if (!videoFlag) {
            Log.w(TAG, Trace.method("create").msg("File not exists"));
            return new Video(videoPath, videoSize, videoWidth, videoHeight, videoDuration,
                    coverPath, coverWidth, coverHeight);
        }

        if (videoFile.length() <= 0L) {
            Log.w(TAG, Trace.method("create").msg("File size is zero"));
            return null;
        }

        MediaMetadataRetriever retriever = new MediaMetadataRetriever();
        String durationString = null;
        String videoWidthStr = null;
        String videoHeightStr = null;
        try {
            retriever.setDataSource(videoPath);
            durationString = retriever.extractMetadata(MediaMetadataRetriever.METADATA_KEY_DURATION);
            videoWidthStr = retriever.extractMetadata(MediaMetadataRetriever.METADATA_KEY_VIDEO_WIDTH);
            videoHeightStr = retriever.extractMetadata(MediaMetadataRetriever.METADATA_KEY_VIDEO_HEIGHT);

        } catch (RuntimeException e) {
            Log.w(TAG, Trace.method("create")
                    .msg(String.format(Locale.US, "Video path: %s, error: %s", videoPath, e.getMessage())));
        } finally {
            // 记得要释放
            retriever.release();
        }

        if (videoWidth < 1 || videoHeight < 1) {
            videoWidth = TextUtils.isEmpty(videoWidthStr) ? 0 : Integer.parseInt(videoWidthStr);
            videoHeight = TextUtils.isEmpty(videoHeightStr) ? 0 : Integer.parseInt(videoHeightStr);
        }

        if (videoDuration < 1) {
            videoDuration = TextUtils.isEmpty(durationString) ? 0 : Integer.parseInt(durationString);
        }

        if (videoSize < 1) {
            videoSize = videoFile.length();
        }

        return new Video(videoPath, videoSize, videoWidth, videoHeight, videoDuration,
                coverPath, coverWidth, coverHeight);
    }

    @Override
    public Error validContent() {
        if (TextUtils.isEmpty(this.videoUrl)) {
            return new Error(ErrorEnum.INVALID_PARAMETER, INVALID_PARAMETER);
        }
        if (this.coverHeight < 0
                || this.coverWidth < 0
                || this.videoWidth < 0
                || this.videoHeight < 0
                || this.videoSize < 0
                || this.videoDuration < 0) {
            return new Error(ErrorEnum.INVALID_PARAMETER, INVALID_PARAMETER);
        }

        if (FileUtils.isNotWebUrl(this.videoUrl)) {
            File file = new File(this.videoUrl);
            if (file.length() > FIFTY_MB) {
                return new Error(ErrorEnum.MSG_SIZE_LIMIT_EXCEEDED, CONTENT_SIZE_LIMIT);
            }
            if (!FileUtils.validVideoType(this.videoUrl)) {
                return new Error(ErrorEnum.INVALID_PARAMETER, CONTENT_FORMAT_ERROR);
            }
        }

        if (!TextUtils.isEmpty(coverUrl) && FileUtils.isNotWebUrl(this.coverUrl)) {
            File file = new File(this.coverUrl);
            if (file.length() > TWENTY_MB) {
                return new Error(ErrorEnum.MSG_SIZE_LIMIT_EXCEEDED, CONTENT_SIZE_LIMIT);
            }
            if (!FileUtils.validImageType(file)) {
                return new Error(ErrorEnum.INVALID_PARAMETER, CONTENT_FORMAT_ERROR);
            }
        }

        return null;
    }

    public String getVideoUrl() {
        return videoUrl;
    }

    public long getVideoSize() {
        return videoSize;
    }

    public int getVideoDuration() {
        return videoDuration;
    }

    public int getVideoWidth() {
        return videoWidth;
    }

    public int getVideoHeight() {
        return videoHeight;
    }

    public String getCoverUrl() {
        return coverUrl;
    }

    public String getCoverThumbUrl() {
        return coverThumbUrl;
    }

    public void setCoverThumbUrl(String coverThumbUrl) {
        this.coverThumbUrl = coverThumbUrl;
    }

    public int getCoverWidth() {
        return coverWidth;
    }

    public int getCoverHeight() {
        return coverHeight;
    }

    @Override
    public String toString() {
        String toStr = "videoUrl='" + videoUrl + '\'' +
                ", videoSize=" + videoSize +
                ", videoDuration=" + videoDuration +
                ", videoWidth=" + videoWidth +
                ", videoHeight=" + videoHeight +
                ", coverUrl='" + coverUrl + '\'' +
                ", coverThumbUrl='" + coverThumbUrl + '\'' +
                ", coverWidth=" + coverWidth +
                ", coverHeight=" + coverHeight;

        if (getProgress() != null) {
            toStr += ", " + getProgress();
        }
        return String.format(Locale.US, "Video{%s}", toStr);
    }

    public static void registerCodecs() {
        registerCodec(new Codec() {
            @Override
            public int type() {
                return TYPE;
            }

            @Override
            public Class<? extends Content> contentClass() {
                return Video.class;
            }

            static final String KEY_VIDEO_URL = "video_url";
            static final String KEY_VIDEO_SIZE = "video_size";
            static final String KEY_VIDEO_DURATION = "video_duration";
            static final String KEY_VIDEO_WIDTH = "video_width";
            static final String KEY_VIDEO_HEIGHT = "video_height";
            static final String KEY_COVER_URL = "cover_url";
            static final String KEY_COVER_THUMB = "cover_thumb";
            static final String KEY_COVER_WIDTH = "cover_width";
            static final String KEY_COVER_HEIGHT = "cover_height";

            @Override
            public String makeDBString(Content content) throws Throwable {
                Video video = (Video) content;
                JSONObject json = new JSONObject();
                json.put(KEY_VIDEO_URL, video.videoUrl);
                json.put(KEY_VIDEO_SIZE, video.videoSize);
                json.put(KEY_VIDEO_DURATION, video.videoDuration);
                json.put(KEY_VIDEO_WIDTH, video.videoWidth);
                json.put(KEY_VIDEO_HEIGHT, video.videoHeight);
                json.put(KEY_COVER_URL, video.coverUrl);

                if (!TextUtils.isEmpty(video.coverThumbUrl)) {
                    json.put(KEY_COVER_THUMB, video.coverThumbUrl);
                }

                json.put(KEY_COVER_WIDTH, video.coverWidth);
                json.put(KEY_COVER_HEIGHT, video.coverHeight);

                return json.toString();
            }

            @Override
            public Content makeChatContent(String data) throws Throwable {
                JSONObject json = new JSONObject(data);

                Video video = new Video(json.getString(KEY_VIDEO_URL),
                        json.getInt(KEY_VIDEO_SIZE),
                        json.getInt(KEY_VIDEO_DURATION),
                        json.getInt(KEY_VIDEO_WIDTH),
                        json.getInt(KEY_VIDEO_HEIGHT),
                        json.getString(KEY_COVER_URL),
                        json.getInt(KEY_COVER_WIDTH),
                        json.getInt(KEY_COVER_HEIGHT));

                if (json.has(KEY_COVER_THUMB)) {
                    video.coverThumbUrl = json.getString(KEY_COVER_THUMB);
                }

                return video;
            }

            @Override
            public String toString() {
                return "VideoCodec";
            }
        });
    }
}
