package com.hummer.im.model.chat.contents;

import android.graphics.BitmapFactory;
import android.text.TextUtils;
import android.util.Patterns;
import android.webkit.URLUtil;

import androidx.annotation.NonNull;

import com.hummer.im.Error;
import com.hummer.im.ErrorEnum;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.Trace;
import com.hummer.im._internals.utility.FileUtils;
import com.hummer.im.model.chat.Content;

import org.json.JSONObject;

import java.io.File;
import java.util.Locale;

public class Image extends Content {

    public static final int TYPE = 1;

    private final int width;

    private final int height;

    private final String originUrl;

    private String thumbUrl;

    private static final String TAG = "Image";

    private Image(String path, int width, int height) {
        this.originUrl = path;
        this.width = width;
        this.height = height;
    }

    public static Image create(@NonNull String path, int width, int height) {
        Log.i(TAG, Trace.method("create")
                .info("path", path)
                .info("width", width)
                .info("height", height));

        // 如果是网络图片，则直接返回
        if (Patterns.WEB_URL.matcher(path).matches() || URLUtil.isValidUrl(path)) {
            Log.w(TAG, Trace.method("createImage").msg("Create url Image"));
            return new Image(path, width, height);
        }

        // 图片不存在则 直接返回
        if (!FileUtils.exists(path)) {
            Log.w(TAG, Trace.method("createImage").msg("Image not exists"));
            return new Image(path, width, height);
        }

        // 宽高 < 1，则由平台解析宽高
        if (width < 1 || height < 1) {
            BitmapFactory.Options options = new BitmapFactory.Options();
            try {
                // 避免全量解码耗费资源
                options.inJustDecodeBounds = true;

                BitmapFactory.decodeFile(path, options);

            } catch (RuntimeException e) {
                Log.w(TAG, Trace.method("createImage")
                        .msg(String.format(Locale.US, "Image decode path: %s, error: %s", path, e.getMessage())));
            }

            return new Image(path, options.outWidth, options.outHeight);
        }

        return new Image(path, width, height);
    }

    @Override
    public Error validContent() {
        if (TextUtils.isEmpty(this.originUrl)) {
            return new Error(ErrorEnum.INVALID_PARAMETER, INVALID_PARAMETER);
        }
        if (this.width < 0 || this.height < 0) {
            return new Error(ErrorEnum.INVALID_PARAMETER, INVALID_PARAMETER);
        }

        if (FileUtils.isNotWebUrl(this.originUrl)) {
            File file = new File(this.originUrl);
            if (file.length() > TWENTY_MB) {
                return new Error(ErrorEnum.MSG_SIZE_LIMIT_EXCEEDED, CONTENT_SIZE_LIMIT);
            }
            if (!FileUtils.validImageType(file)) {
                return new Error(ErrorEnum.INVALID_PARAMETER, CONTENT_FORMAT_ERROR);
            }
        }

        return null;
    }

    public int getWidth() {
        return width;
    }

    public int getHeight() {
        return height;
    }

    public String getOriginUrl() {
        return originUrl;
    }

    public String getThumbUrl() {
        return thumbUrl;
    }

    public void setThumbUrl(String thumbUrl) {
        this.thumbUrl = thumbUrl;
    }

    @Override
    public String toString() {
        String toStr = "width=" + width +
                ", height=" + height +
                ", originUrl='" + originUrl + '\'' +
                ", thumbUrl='" + thumbUrl + '\'';

        if (getProgress() != null) {
            toStr += ", " + getProgress();
        }
        return String.format(Locale.US, "Image{%s}", toStr);
    }

    public static void registerCodecs() {
        registerCodec(new Codec() {
            @Override
            public int type() {
                return TYPE;
            }

            @Override
            public Class<? extends Content> contentClass() {
                return Image.class;
            }

            static final String KeyWidth    = "width";
            static final String KeyHeight   = "height";
            static final String KeyOrigin   = "origin";
            static final String KeyThumb    = "thumb";

            @Override
            public String makeDBString(Content content) throws Throwable {
                Image image = (Image) content;

                JSONObject json = new JSONObject();
                json.put(KeyWidth,    image.width);
                json.put(KeyHeight,   image.height);
                json.put(KeyOrigin,   image.originUrl);

                boolean flag = image.thumbUrl == null || image.thumbUrl.length() == 0;
                if (!flag) {
                    json.put(KeyThumb,    image.thumbUrl);
                }
                return json.toString();
            }

            @Override
            public Content makeChatContent(String data) throws Throwable {
                JSONObject json = new JSONObject(data);

                Image image = new Image(
                        json.getString(KeyOrigin),
                        json.getInt(KeyWidth),
                        json.getInt(KeyHeight)
                );

                if (json.has(KeyThumb)) {
                    image.thumbUrl = json.getString(KeyThumb);
                }

                return image;
            }

            @Override
            public String toString() {
                return "ImageCodec";
            }
        });
    }
}
