package com.hummer.im.model.chat.contents;

import android.media.MediaMetadataRetriever;
import android.text.TextUtils;

import com.hummer.im.Error;
import com.hummer.im.ErrorEnum;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.Trace;
import com.hummer.im._internals.utility.FileUtils;
import com.hummer.im.model.chat.Content;

import org.json.JSONObject;

import java.io.File;
import java.util.Locale;

public class Audio extends Content {

    public static final int TYPE = 3;

    /**
     * 音频路径
     */
    private final String url;

    /**
     * 音频时长，单位毫秒
     */
    private final int duration;

    private static final String TAG = "Audio";

    private Audio(String url, int duration) {
        this.url = url;
        this.duration = duration;
    }

    public static Audio create(String path, int duration) {
        Log.i(TAG, Trace.method("create")
                .info("path", path)
                .info("duration", duration));

        if (FileUtils.isWebUrl(path)) {
            Log.w(TAG, Trace.method("create").msg("Create url Audio"));
            return new Audio(path, duration);
        }

        // 不存在，直接返回
        if (!FileUtils.exists(path)) {
            Log.w(TAG, Trace.method("create").msg("Audio not exists"));
            return new Audio(path, duration);
        }

        if (duration < 1) {
            MediaMetadataRetriever retriever = new MediaMetadataRetriever();
            try {
                retriever.setDataSource(path);
                String durationString = retriever.extractMetadata(MediaMetadataRetriever.METADATA_KEY_DURATION);
                if (durationString != null && !TextUtils.isEmpty(durationString)) {
                    duration = Integer.parseInt(durationString);
                }
            } catch (RuntimeException e) {
                Log.w(TAG, Trace.method("create")
                        .msg(String.format(Locale.US, "audio path: %s, error: %s", path, e.getMessage())));
                duration = 0;
            } finally {
                retriever.release();// 记得要释放
            }

            return new Audio(path, duration);
        }

        return new Audio(path, duration);
    }

    @Override
    public Error validContent() {
        if (TextUtils.isEmpty(this.url)) {
            return new Error(ErrorEnum.INVALID_PARAMETER, INVALID_PARAMETER);
        }
        if (this.duration < 0) {
            return new Error(ErrorEnum.INVALID_PARAMETER, INVALID_PARAMETER);
        }

        if (FileUtils.isNotWebUrl(this.url)) {
            File file = new File(this.url);
            if (file.length() > TWENTY_MB) {
                return new Error(ErrorEnum.MSG_SIZE_LIMIT_EXCEEDED, CONTENT_SIZE_LIMIT);
            }
            if (!FileUtils.validAudioType(this.url)) {
                return new Error(ErrorEnum.INVALID_PARAMETER, CONTENT_FORMAT_ERROR);
            }
        }

        return null;
    }

    public String getUrl() {
        return url;
    }

    public int getDuration() {
        return duration;
    }

    @Override
    public String toString() {
        String toStr = "url='" + url + '\'' +
                ", duration=" + duration;

        if (getProgress() != null) {
            toStr += ", " + getProgress();
        }

        return String.format(Locale.US, "Audio{%s}", toStr);
    }

    /**
     * 语音消息内容的编码器、解码器注册方法
     */
    public static void registerCodecs() {
        registerCodec(new Codec() {
            @Override
            public int type() {
                return TYPE;
            }

            @Override
            public Class<? extends Content> contentClass() {
                return Audio.class;
            }

            private static final String KeyUrl      = "url";
            private static final String KeyDuration = "dur";

            @Override
            public String makeDBString(Content content) throws Throwable {
                Audio audio = (Audio) content;

                JSONObject json = new JSONObject();
                json.put(KeyUrl,      audio.url);
                json.put(KeyDuration, audio.duration);

                return json.toString();
            }

            @Override
            public Content makeChatContent(String data) throws Throwable {
                JSONObject json = new JSONObject(data);
                return new Audio(json.getString(KeyUrl), json.getInt(KeyDuration));
            }

            @Override
            public String toString() {
                return "AudioCodec";
            }
        });
    }
}
