package com.hummer.im.model.chat;

import android.text.TextUtils;
import android.util.Base64;

import com.hummer.im._internals.log.Log;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * 聊天消息携带的推送消息内容
 */
@SuppressWarnings({"WeakerAccess", "unused"})
public class PushContent {

    /**
     * 推送消息的标题内容
     */
    public String getTitle() {
        return title;
    }

    /**
     * 推送消息的消息内容
     */
    public String getContent() {
        return content;
    }

    /**
     * payload 是推送消息的业务携带数据
     */
    public byte[] getPayload() {
        return payload;
    }

    /**
     * 通知栏中展示的图标链接
     */
    public String getIconUrl() {
        return iconUrl;
    }

    /**
     * 构造消息推送所需的Content实例，用于帮助CIM后台确定消息推送的具体内容。
     *
     * @param title   消息推送的标题内容
     * @param content 消息推送的消息内容
     * @param payload 消息推送所携带的业务自定义数据
     * @param iconUrl 消息推送弹通知栏时所显示的图标URL字符串
     */
    public PushContent(String title, String content, byte[] payload, String iconUrl) {
        this.title   = title;
        this.content = content;
        this.payload = payload;
        this.iconUrl = iconUrl;
    }

    public static String makeString(PushContent pushContent) {
        if (pushContent == null) {
            return "";
        }

        JSONObject json = new JSONObject();
        try {
            json.put(FIELD_TITLE, pushContent.getTitle());
            json.put(FIELD_CONTENT, pushContent.getContent());
            json.put(FIELD_PAYLOAD, Base64.encodeToString(pushContent.getPayload(), Base64.DEFAULT));
            json.put(FIELD_ICON_URL, pushContent.getIconUrl());

            return json.toString();
        } catch (JSONException e) {
//            Log.e(TAG, Trace.method("makeString").info("push content to json error", e.getMessage()));
            return "";
        }
    }

    public static PushContent makeContent(String jsonStr) {
        if (TextUtils.isEmpty(jsonStr)) {
            return null;
        }

        try {
            JSONObject json = new JSONObject(jsonStr);

            return new PushContent(json.getString(FIELD_TITLE),
                    json.getString(FIELD_CONTENT),
                    Base64.decode(json.getString(FIELD_PAYLOAD), Base64.DEFAULT),
                    json.getString(FIELD_ICON_URL));
        } catch (JSONException e) {
//            Log.e(TAG, Trace.method("makeContent").info("json to push content error", e.getMessage()));
            return null;
        }
    }

    @Override
    public String toString() {
        return "PushContent{" + new StringChain()
                .add("title",   title)
                .add("cont",    content)
                .add("iconUrl", iconUrl)
                + "}";
    }

    private final String title;
    private final String content;
    private final byte[] payload;
    private final String iconUrl;

    private static final String FIELD_TITLE = "title";
    private static final String FIELD_CONTENT = "content";
    private static final String FIELD_PAYLOAD = "payload";
    private static final String FIELD_ICON_URL = "icon_url";

    private static final String TAG = "PushContent";
}
