package com.hummer.im.model.chat;

import androidx.annotation.NonNull;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.bridge.helper.HummerEngine;
import com.hummer.im.model.chat.states.Archived;
import com.hummer.im.model.chat.states.Delivering;
import com.hummer.im.model.chat.states.Failed;
import com.hummer.im.model.chat.states.Init;
import com.hummer.im.model.chat.states.Preparing;
import com.hummer.im.model.chat.store.MessageStoreStrategy;
import com.hummer.im.model.id.Identifiable;
import com.hummer.im.model.id.User;

import java.util.Map;

/**
 * ChatMessage是Hummer服务的核心类型，用于表示一个用户消息的具体实例。它包含：
 * <br> 1. 用于识别消息来源地、目的地的sender、receiver属性
 * <br> 2. 用于承载消息内容的chatContent属性
 * <br> 3. 用于承载消息推送服务元信息的pushContent属性
 * <br> 4. 用于表示消息处理状态的state属性
 * <br> 5. 其它消息相关的属性，如时间戳(timestamp), 唯一识别uuid
 */
@SuppressWarnings({"unused", "WeakerAccess"})
public final class Message {

    /**
     * 构造一个空的ChatMessage对象
     */
    public Message() {
        this.uuid = HummerEngine.makeUUID();
        this.timestamp = HummerEngine.getSyncTs();
        this.state = new Init();
    }

    /**
     * ChatMessage的状态抽象概念
     */
    public interface State {
    }

    /**
     * 从jni层回来的状态值构建native层的状态类
     *
     * @param enumValue
     * @param code
     * @param desc
     *
     * @return
     */
    public static State buildSate(int enumValue, int code, String desc) {
        State state = new Init();
        switch (enumValue) {
            case 1:
                state = new Preparing();
                break;
            case 2:
                state = new Delivering();
                break;
            case 3:
                state = new Archived();
                break;
            case 4:
                state = new Failed(new Error(code, desc));
                break;
            default:
                break;
        }
        return state;
    }

    public static State buildSate(int enumValue, int code, String desc, Map<String, String> extraInfo) {
        State state = new Init();
        switch (enumValue) {
            case 1:
                state = new Preparing();
                break;
            case 2:
                state = new Delivering();
                break;
            case 3:
                state = new Archived();
                break;
            case 4:
                state = new Failed(new Error(code, desc, extraInfo));
                break;
            default:
                break;
        }
        return state;
    }

    /**
     * 构造一个预设了消息接收者和消息内容的ChatMessage对象
     *
     * @param receiver 消息接收的目标
     * @param content  消息内容
     */
    public Message(Identifiable receiver, Content content) {
        this(receiver, content, new MessageStoreStrategy());
    }

    /**
     * 构造器
     *
     * @param receiver      接收者
     * @param content       消息内容
     * @param storeStrategy 消息存储策略
     */
    public Message(Identifiable receiver, Content content, MessageStoreStrategy storeStrategy) {
        this.chatContent = content;
        this.receiver = receiver;
        this.storeStrategy = storeStrategy;
        this.timestamp = HummerEngine.getSyncTs();
        this.uuid = HummerEngine.makeUUID();
        this.state = new Init();
    }

    /**
     * 获取消息内容
     */
    public Content getContent() {
        return this.chatContent;
    }

    /**
     * 设置消息内容
     */
    public void setContent(Content content) {
        this.chatContent = content;
    }

    /**
     * 每个ChatMessage实例，都可以通过设置PushContent属性来附加消息推送的能力，Hummer服务器在接收到携带了PushContent
     * 内容的消息时，向接收者发起系统推送消息。该推送消息的具体内容，由PushContent所包含的内容来决定。
     * 如果该属性为null，则表示该消息不具备触发消息推送的功能。
     */
    public PushContent getPushContent() {
        return pushContent;
    }

    /**
     * 设置消息携带的推送内容
     */
    public void setPushContent(PushContent pushContent) {
        this.pushContent = pushContent;
    }

    /**
     * 消息的发送者，由于是聊天消息(Message)，其产生源一定是用户，因此其类型只可能是User
     * 如果是user实例，在进行会话界面渲染时，可以通过HMR.isMe()来快速判断该消息是否由当前用户所发送。
     */
    public Identifiable getSender() {
        return sender;
    }

    /**
     * 设置消息的发送方。如果是本地发送出去的消息，会在调用ChatService.send方法时被自动覆盖为本地用户(Me)
     *
     * @param sender 消息的发送者
     */
    public void setSender(Identifiable sender) {
        this.sender = sender;
    }

    /**
     * 消息的接收者，具体类型是多态的，可能为Person(用户聊天会话)，Group(群组聊天会话)……
     */
    public Identifiable getReceiver() {
        return receiver;
    }

    /**
     * 设置消息接收者。对于本地发出的消息，业务必须显示为ChatMessage显示设置Receiver属性。
     *
     * @param receiver 消息接收者
     */
    public void setReceiver(Identifiable receiver) {
        this.receiver = receiver;
    }

    /**
     * 对于当前用户来说，该方法用于返回聊天对应的目标对象。
     *
     * @return 如果是P2P消息，返回当前用户以外的用户对象。如果是群、聊天室、官方号等消息，返回receiver作为target
     */
    public Identifiable getTarget() {
        // 对于P2P聊天消息，判断会话目标的逻辑表如下：
        // 1. sender: me,     receiver: me          <- receiver
        // 2. sender: me,     receiver: fellow      <- receiver
        // 3. sender: fellow, receiver: me          <- sender
        // 4. sender: fellow, receiver: fellow      <- N/A 我方不该收到该消息
        if (sender instanceof User && receiver instanceof User) {
            if (HMR.isMe(sender)) {
                return receiver;
            } else if (HMR.isMe(receiver)) {
                return sender;
            } else {
                return null;
            }
        }
        // 对于如群消息、聊天室消息、公众号消息等非P2P消息，receiver一定是该关系对象本身
        // sender为发送消息到该关系的人，因此可以直接返回receiver
        return receiver;
    }

    /**
     * timestamp属性为消息时间戳，在消息创建时，该时间戳为即为Hummer服务器同步时间戳，发送（将发送）时覆盖成发送时间戳
     * 考虑到本地时间和服务器时间可能存在时差，以及服务器集群中，各服务器间的时间存在时间差等原因，该时间戳
     * 只能作为近似（绝大部分情况下）判别消息先后顺序的依据。
     */
    public long getTimestamp() {
        return this.timestamp;
    }

    /**
     * 设置消息的时间戳
     */
    public void setTimestamp(long ts) {
        this.timestamp = ts;
    }

    /**
     * uuid为消息的（近似）唯一标识id，由于uuid是在各个客户端本地独立产生的，因此理论上存在碰撞的可能性。但由于这
     * 种概率极其低，因此可以用作消息的唯一识别标识，对消息进行去重等操作。
     */
    public String getUuid() {
        return this.uuid;
    }

    /**
     * 设置消息唯一标识符，该标识符通常是由Hummer内部产生并设置的，业务一般不调用该方法
     */
    public void setUuid(String id) {
        this.uuid = id;
    }

    /**
     * 每个ChatMessage对象都可以附带一个业务扩展信息。该信息可以帮助业务透传例如用户头像等业务逻辑紧密相关的数据。Hummer只
     * 对该数据进行透传，不做具体解释和处理。
     */
    public String getAppExtra() {
        return appExtra;
    }

    /**
     * 设置消息的业务透传数据
     */
    public void setAppExtra(String appExtra) {
        this.appExtra = appExtra;
    }

    public Map<String, String> getKvExtra() {
        return kvExtra;
    }

    public void setKvExtra(Map<String, String> kvExtra) {
        this.kvExtra = kvExtra;
    }

    /**
     * state 属性用于标识当前Message对象所处的具体状态。在进行界面渲染时，应通过RTTI来进行类型判别，并针对不同的状态
     * 进行相应的渲染处理。
     * <p>
     * {@link Init}
     * {@link Preparing}
     * {@link Delivering}
     * {@link Archived}
     * {@link Failed}
     *
     * @return ChatMessage的当前状态
     */
    public State getState() {
        return state;
    }

    /**
     * 设置ChatMessage的当前状态，设置状态时
     */
    public void setState(@NonNull State newState) {
        if (newState == null) {
//            Log.e("Source", Trace.method("performStarting").msg("newState is <null>"));
            return;
        }

        state = newState;
    }

    public MessageStoreStrategy getStoreStrategy() {
        return storeStrategy;
    }

    public void setStoreStrategy(MessageStoreStrategy storeStrategy) {
        this.storeStrategy = storeStrategy;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Message)) {
            return false;
        }
        Message that = (Message) o;

        return (this.uuid != null) && (that.uuid != null) && uuid.equals(that.uuid);
    }

    @Override
    public int hashCode() {
        return uuid.hashCode();
    }

    @Override
    public String toString() {
        return "Message{" +
                "uuid='" + uuid + '\'' +
                '}';
    }

    private static final String TAG = "Message";

    private String uuid;
    private Identifiable sender;
    private Identifiable receiver;
    private PushContent pushContent;
    private Long timestamp;
    private State state;
    private String appExtra;
    private Map<String, String> kvExtra;
    private Content chatContent;

    /**
     * 消息存储策略
     */
    private MessageStoreStrategy storeStrategy = new MessageStoreStrategy();
}
