package com.hummer.im.model.chat;

import android.util.Base64;

import com.hummer.im.Error;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.Trace;
import com.hummer.im.model.chat.contents.Text;

/**
 * ChatContent用于表示一个聊天内容类型，具体类型应该由其子类来实现。例如Text, Voice, Image等
 *
 * @see Text
 */
public class Content {

    /**
     * Hummer支持Text, Image, Audio等内置消息内容，它们的type值取值范围均小于 CustomTypeStart 值，业务如果需要
     * 实现自定义消息内容，应保证其type值应大于等于CustomTypeStart值，否则，自定义类型注册将会断言失败。
     */
    @SuppressWarnings({"WeakerAccess", "unused"})
    public final static int CustomTypeStart = 10000;

    private ContentProgress progress;

    public ContentProgress getProgress() {
        return progress;
    }

    public void setProgress(ContentProgress progress) {
        this.progress = progress;
    }

    /**
     * 内容合法校验
     *
     * @return Error
     */
    public Error validContent(){
        return null;
    };

    /**
     * 根据消息内容获取其类型值
     */
    public static Integer getDataType(Content content) {
        if (content instanceof AppContent) {
            return ((AppContent) content).getType();
        }

        return dbCodecs.getDataType(content);
    }

    /**
     * 消息内容的本地存储数据序列化方法
     */
    public static String makeString(Content content) {
        if (content instanceof AppContent) {
            return Base64.encodeToString(((AppContent) content).getData(), Base64.DEFAULT);
        }

        return dbCodecs.encode(content);
    }

    public static Content makeContent(int type, String string) {
        if (type >= CustomTypeStart && dbCodecs.isNotExistCodec(type)) {
            return new AppContent(type, Base64.decode(string, Base64.DEFAULT));
        }

        return dbCodecs.decode(type, string);
    }

    public static Content makeStringContent(int type, String string) {
        if (type >= CustomTypeStart && dbCodecs.isNotExistCodec(type)) {
            return new AppContent(type, string.getBytes());
        }

        return dbCodecs.decode(type, string);
    }

    /**
     * 所有的Hummer消息都需要支持对应的内容编码、解码器，如果需要实现业务自定义消息类型，则需要实现下述Codec接口，并在业务初始化时
     * 调用registerCodec方法进行注册，否则Hummer无法正确处理该类型的消息。
     */
    public interface Codec {
        /**
         * 消息的类型标识值，业务在实现自定义消息时，应使用大于等于CustomTypeStart(10000)的取值，避免和Hummer内置类型冲突。该值
         * 一般用于消息内容的反序列化。
         */
        int type();

        /**
         * 消息类型的类对象
         */
        Class<? extends Content> contentClass();

        /**
         * 构造用于本地数据库存储的字符串内容
         *
         * @param content 消息内容对象
         */
        String makeDBString(Content content) throws Throwable;

        Content makeChatContent(String data) throws Throwable;
    }

    /**
     * 注册业务自定义消息类型的工厂对象
     *
     * @param codec 用于具体消息的编解码器
     */
    protected static void registerCodec(final Codec codec) {
        Log.i("Content", Trace.method("registerCodec")
                .msg("registerCodec")
                .info("type", codec.type()).info("codec", codec));

        dbCodecs.register(new CodecManager.Codec<Integer, String, Content>() {
            @Override
            public Integer getDataType() {
                return codec.type();
            }

            @Override
            public Class<? extends Content> getModelClass() {
                return codec.contentClass();
            }

            @Override
            public String encode(Content content) throws Throwable {
                return codec.makeDBString(content);
            }

            @Override
            public Content decode(String data) throws Throwable {
                return codec.makeChatContent(data);
            }
        });
    }

    private static final class DBCodecs extends CodecManager<Integer, String, Content> {

    }

    private static final DBCodecs dbCodecs = new DBCodecs();

    protected final static String INVALID_PARAMETER = "Invalid Parameter";
    protected final static String CONTENT_SIZE_LIMIT = "Content Size Limit";
    protected final static String CONTENT_FORMAT_ERROR = "Content Format Error";
    protected final static int FIFTY_MB = 50 * 1024 * 1024;
    protected final static int TWENTY_MB = 20 * 1024 * 1024;
}
