package com.hummer.im._internals.utility;

import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.hummer.im.Error;
import com.hummer.im.ErrorEnum;
import com.hummer.im.HMR;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.Trace;
import com.hummer.im.model.RequestId;
import com.hummer.im.model.completion.OnFailure;
import com.hummer.im.model.completion.OnSuccess;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * CIM包含了许多服务接口，例如ChatService, ……但它们的具体实现是通过注册的方式来设置的。
 * Java 6开始，提供SPI的，例如：
 *
 * @see <a href="https://juejin.im/post/5af952fdf265da0b9e652de3">Java SPI机制详解</a>
 * 但是，Android中使用起来较为复杂，因此，这里使用Service类型简单地实现了SPI的能力，让用户可以通过纯接口的
 * 方式获得CIM内部提供的服务，以此实现接口和实现的分离。
 */
public class ServiceProvider {

    private static final String TAG = "ServiceProvider";

    public interface ModuleLoader {
        void initModule();

        void deInitModule();

        Map<Class, Service> getServices();
    }

    public interface Service {

        void initService();

        void openService(@NonNull HMRCompletion completion);

        int serviceSort();

        void closeService();
    }

    public static void openServices(final HMRCompletion completion) {
        final List<Service> sortedServices = sortServices();
        Log.i(TAG, Trace.method("openServices").msg(namesForServices(sortedServices)));
        openServices(sortedServices, 0, completion);
    }

    private static void openServices(
            final List<Service> allServices,
            final int atPosition,
            final HMRCompletion completion) {
        if (isCloseState() || !HMRContext.isCurrentContext(completion.getRequestId())) {
            Log.i(TAG, Trace.method("login1").msg("Terminal by logout")
                    .info("state", HMR.getState())
                    .info("currentContext", HMRContext.contextId == null ? 0 : HMRContext.contextId.getId())
                    .info("requestId", completion.getRequestId().getId()));
            CompletionUtils.dispatchFailure(completion, new Error(ErrorEnum.BAD_USER_ERROR, "Terminal by logout"));
            return;
        }

        if (atPosition >= allServices.size()) {
            CompletionUtils.dispatchSuccess(completion);
            return;
        }

        final Service service = allServices.get(atPosition);
        service.openService(new HMRCompletion(completion.getRequestId())
                .onSuccess(new OnSuccess() {
                    @Override
                    public void onSuccess() {
                        Log.d(TAG,
                                String.format("openService | Success - %s", service.getClass().getSimpleName()));

                        openServices(allServices, atPosition + 1, completion);
                    }
                })
                .onFailure(new OnFailure() {
                    @Override
                    public void onFailure(Error error) {
                        Log.e(TAG, Trace.method("openService")
                                .msg("Failed - %s, error: %s", service.getClass().getSimpleName(), error));

                        // Unwind services
                        for (int pos = atPosition - 1; pos >= 0; --pos) {
                            allServices.get(pos).closeService();
                            Log.i(TAG, Trace.method("[rollback]closeService").msg(allServices.get(pos).getClass().getSimpleName()));
                        }

                        CompletionUtils.dispatchFailure(completion, error);
                    }
                })
        );
    }

    public static void closeServices() {
        final List<Service> sortedServices = sortServices();

        Log.i(TAG, Trace.method("closeServices")
                .msg(namesForServices(sortedServices)));

        // close的顺序应该和open的顺序是镜像对称的
        Collections.reverse(sortedServices);

        for (Service service : sortedServices) {
            service.closeService();

            Log.i(TAG, Trace.method("closeService").msg(service.getClass().getSimpleName()));
        }
    }

    private static List<Service> sortServices() {
        List<Service> serviceList = new ArrayList<>(services.values());
        for (Service service : serviceList) {
            service.serviceSort();
        }

        Collections.sort(serviceList, new Comparator<Service>() {

            @Override
            public int compare(Service o1, Service o2) {
                return o1.serviceSort() - o2.serviceSort();
            }
        });

        return serviceList;
    }

    private static String namesForServices(List<Service> services) {
        List<String> names = new ArrayList<>();
        for (Service svc : services) {
            names.add(svc.getClass().getSimpleName());
        }

        return TextUtils.join(", ", names);
    }

    @SuppressWarnings("WeakerAccess")
    public static void register(Class<?> interfaceClass, Service service) {
        services.put(interfaceClass, service);

    }

    /**
     * 获取Interface类型Service的实现实例，在此之前，必须通过register方法注册了该接口的实例
     *
     * @param serviceClass service的类型，例如ChatService.class
     *
     * @return 如果找到了serviceClass对应的服务，则返回其服务实例
     */
    @SuppressWarnings("unchecked")
    public static <Interface> Interface get(Class<Interface> serviceClass) {

        if (serviceClass == null) {
            Log.e(TAG, Trace.method("get")
                    .info("Exception", "null"));
            return null;
        }

        if (services.isEmpty()) {
            ServiceProvider.loadServicesIfNeeded("com.hummer.im");
        }

        for (Map.Entry<Class, Service> impl : services.entrySet()) {
            if (serviceClass.isAssignableFrom(impl.getKey())) {
                return (Interface) impl.getValue();
            }
        }

        Service service = services.get(serviceClass);

        if (service == null) {
            Log.e(TAG, Trace.method("get")
                    .info("Service Not Found:", serviceClass.getName()));
        }

        return (Interface) service;
    }

    public static void loadServicesIfNeeded(String packageName) {
        if (isLoaded) {
            return;
        }

        Set<Class<?>> loaderClasses = ReflectionExt.loadClasses(
                packageName,
                Object.class,
                new ReflectionExt.Filter() {
                    @Override
                    public boolean shouldAccept(String className) {
                        String simpleName = className.substring(className.lastIndexOf(".") + 1);
                        return simpleName.startsWith("SP") && simpleName.endsWith("ModuleLoader");
                    }
                }
        );

        try {
            registerServicesByModuleLoaders(loaderClasses);
        } catch (Exception e) {
            Log.e(TAG, Trace.method("loadServicesIfNeeded")
                    .msg("Fail loading services" + e.getLocalizedMessage()));
        }

        isLoaded = true;
    }

    private static void registerServicesByModuleLoaders(Set<Class<?>> loaderClasses)
            throws InstantiationException, IllegalAccessException {
        for (Class<?> loaderClazz : loaderClasses) {
            ModuleLoader loader = (ModuleLoader) loaderClazz.newInstance();

            loader.initModule();

            loaders.add(loader);

            Map<Class, Service> services = loader.getServices();
            for (Map.Entry<Class, Service> entry : services.entrySet()) {
                ServiceProvider.register(entry.getKey(), entry.getValue());
                entry.getValue().initService();
            }
        }
    }


    private static boolean isLoaded = false;
    private static final Set<ModuleLoader> loaders = new HashSet<>();
    private static final HashMap<Class, Service> services = new HashMap<>();

    public static boolean isCloseState() {
        return HMR.getState() == HMR.State.Closing || HMR.getState() == HMR.State.Closed;
    }
}
