package com.hummer.im._internals.utility;

import android.os.Handler;
import android.os.Looper;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.Trace;
import com.hummer.im.model.RequestId;
import com.hummer.im.model.completion.OnFailure;
import com.hummer.im.model.completion.OnSuccessArg;

import androidx.annotation.NonNull;

public final class HMRCompletionArg<T> {
    public HMRCompletionArg(final RequestId requestId, final HMR.CompletionArg<T> completion) {
        Looper looper = Looper.myLooper();
        if (looper == Looper.getMainLooper()) {
            this.dispatchQueue = new DispatchQueue(new DispatchQueue.LooperHandlerProvider(Looper.getMainLooper()));
        } else {
            // 默认使用Direct分派
            this.dispatchQueue = DispatchQueue.direct;
        }

        onSuccess(new OnSuccessArg<T>() {
            @Override
            public void onSuccess(T result) {
                if (completion != null) {
                    Log.i(TAG, Trace.method("RichCompletionArg").msg("Success")
                            .info("hummerRequestId", requestId));
                    completion.onSuccess(result);
                }
            }
        }).onFailure(new OnFailure() {
            @Override
            public void onFailure(Error error) {
                if (completion != null) {
                    Log.i(TAG, Trace.method("RichCompletionArg").msg("Fail")
                            .info("hummerRequestId", requestId)
                            .info("Error", error.toString()));
                    completion.onFailed(error);
                }
            }
        });
    }

    public HMRCompletionArg(@NonNull Handler handler) {
        this.dispatchQueue = new DispatchQueue(new DispatchQueue.LooperHandlerProvider(handler.getLooper()));
    }

    public HMRCompletionArg<T> onSuccess(@NonNull OnSuccessArg<T> success) {
        this.successHandler = success;
        return this;
    }

    public HMRCompletionArg<T> onFailure(@NonNull OnFailure failure) {
        this.failureHandler = failure;
        return this;
    }

    void dispatchSuccess(@NonNull T result) {

        final T finalResult = result;
        dispatchQueue.async(new Runnable() {
            @Override
            public void run() {
                if (successHandler != null) {
                    successHandler.onSuccess(finalResult);
                } else {
                    Log.i(TAG, Trace.method("dispatchSuccess")
                            .msg("dispatchQueue async, success handler = null"));
                }
            }
        });
    }

    void dispatchFailure(@NonNull Error error) {
        final Error finalError = error;
        dispatchQueue.async(new Runnable() {
            @Override
            public void run() {
                if (failureHandler != null) {
                    failureHandler.onFailure(finalError);
                } else {
                    Log.i(TAG, Trace.method("dispatchFailure")
                            .msg("dispatchQueue async, failure handler = null"));
                }
            }
        });
    }

    private static final String TAG = "RichCompletionArg";
    private final DispatchQueue dispatchQueue;
    private OnFailure failureHandler;
    private OnSuccessArg<T> successHandler;
}
