package com.hummer.im._internals.utility;

import android.os.Handler;
import android.os.Looper;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.Trace;
import com.hummer.im.model.RequestId;
import com.hummer.im.model.completion.OnFailure;
import com.hummer.im.model.completion.OnSuccess;

import java.util.LinkedList;
import java.util.List;

public final class HMRCompletion {

    public HMRCompletion() {
        Looper looper = Looper.myLooper();
        if (looper == Looper.getMainLooper()) {
            this.dispatchQueue = new DispatchQueue(new DispatchQueue.LooperHandlerProvider(Looper.getMainLooper()));
        } else if (looper == HMRContext.work.getHandler().getLooper()) {
            // 回到work队列
            this.dispatchQueue = HMRContext.work;
        } else {
            // 默认使用Direct分派
            this.dispatchQueue = DispatchQueue.direct;
        }
    }

    public HMRCompletion(@NonNull final RequestId requestId) {
        this();
        this.requestId = requestId;
    }

    public HMRCompletion(@NonNull final RequestId requestId, @Nullable final HMR.Completion completion) {
        this();
        this.requestId = requestId;

        this.onSuccess(new OnSuccess() {
            @Override
            public void onSuccess() {
                if (completion != null) {
                    Log.i(TAG, Trace.method("HMRCompletion").msg("Success")
                            .info("hummerRequestId", requestId));
                    completion.onSuccess();
                }
            }
        }).onFailure(new OnFailure() {
            @Override
            public void onFailure(Error error) {
                if (completion != null) {
                    Log.i(TAG, Trace.method("HMRCompletion").msg("Fail")
                            .info("hummerRequestId", requestId)
                            .info("Error", error.toString()));
                    completion.onFailed(error);
                }
            }
        });
    }

    public HMRCompletion(@NonNull Handler handler) {
        this.dispatchQueue = new DispatchQueue(new DispatchQueue.LooperHandlerProvider(handler.getLooper()));
    }

    public HMRCompletion decorate(@Nullable HMRCompletion decorated) {
        if (this.decorated == null) {
            Log.e(TAG, Trace.method("decorate")
                    .info("只允许对Completion进行一次装饰", this.decorated));
            return this;
        }

        this.decorated = decorated;
        return this;
    }

    public HMRCompletion beforeSuccess(@NonNull Runnable runnable) {
        this.preSuccess.add(runnable);
        return this;
    }

    public HMRCompletion onSuccess(@NonNull OnSuccess success) {
        this.successHandler = success;
        return this;
    }

    public HMRCompletion afterSuccess(@NonNull Runnable runnable) {
        this.postSuccess.add(0, runnable);
        return this;
    }

    public HMRCompletion beforeFailure(@NonNull Runnable runnable) {
        this.preFailure.add(runnable);
        return this;
    }

    public HMRCompletion onFailure(@NonNull OnFailure failure) {
        this.failureHandler = failure;
        return this;
    }

    public HMRCompletion afterFailure(@NonNull Runnable runnable) {
        this.postFailure.add(0, runnable);
        return this;
    }

    void dispatchSuccess() {
        dispatchQueue.async(new Runnable() {
            @Override
            public void run() {
                dispatchActions(preSuccess);

                if (successHandler != null) {
                    successHandler.onSuccess();
                } else {
                    Log.i(TAG, Trace.method("dispatchSuccess")
                            .msg("dispatchSuccess, successHandler == null"));
                }

                CompletionUtils.dispatchSuccess(decorated);

                dispatchActions(postSuccess);
            }
        });
    }

    void dispatchFailure(@NonNull final Error error) {
        dispatchQueue.async(new Runnable() {
            @Override
            public void run() {
                dispatchActions(preFailure);

                if (failureHandler != null) {
                    failureHandler.onFailure(error);
                } else {
                    Log.i(TAG, Trace.method("dispatchFailure")
                            .msg("dispatchFailure, failureHandler == null"));
                }

                CompletionUtils.dispatchFailure(decorated, error);

                dispatchActions(postFailure);
            }
        });
    }

    private static void dispatchActions(@NonNull List<Runnable> actions) {
        for (Runnable r : actions) {
            r.run();
        }
    }

    private static final String TAG = "HMRCompletion";

    private final DispatchQueue dispatchQueue;
    private List<Runnable> preSuccess = new LinkedList<>();
    private OnSuccess successHandler;
    private List<Runnable> postSuccess = new LinkedList<>();

    private List<Runnable> preFailure = new LinkedList<>();
    private OnFailure failureHandler;
    private List<Runnable> postFailure = new LinkedList<>();

    private HMRCompletion decorated;

    public RequestId getRequestId() {
        return requestId;
    }

    private RequestId requestId = new RequestId(0);
}
