package com.hummer.im._internals.utility;

import android.text.TextUtils;
import android.util.Patterns;
import android.webkit.URLUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

public class FileUtils {

    /**
     * 判断文件是否存在
     *
     * @param path 文件路径
     *
     * @return 存在: true; 不存在: false
     */
    public static boolean exists(String path) {
        if (TextUtils.isEmpty(path)){
            return false;
        }

        File file = new File(path);

        return file.exists() && !file.isDirectory();
    }

    /**
     * 判断url是否是网络url
     *
     * @param url url
     *
     * @return true: 是; false: 否
     */
    public static boolean isWebUrl(String url) {
        return Patterns.WEB_URL.matcher(url).matches() || URLUtil.isValidUrl(url);
    }

    public static boolean isNotWebUrl(String url) {
        return !isWebUrl(url);
    }

    public static boolean validImageType(File imageFile) {
        String header = getFileHeader(imageFile);
        if (TextUtils.isEmpty(header)) {
            return false;
        }

        return header.contains(FileType.JPG.getHeader())
                || header.contains(FileType.PNG.getHeader())
                || header.contains(FileType.BMP.getHeader());
    }

    public static boolean validAudioType(String audioPath) {
        return validFileSuffix(audioPath, "aac");
    }

    public static boolean validVideoType(String videoPath) {
        return validFileSuffix(videoPath, "mp4");
    }

    private static boolean validFileSuffix(String path, String validSuffix) {
        if (TextUtils.isEmpty(validSuffix)) {
            return true;
        }

        if (TextUtils.isEmpty(path)) {
            return false;
        }

        int index = path.lastIndexOf(".");
        if (index > 0) {
            String fileSuffix = path.substring(index + 1);
            return !TextUtils.isEmpty(fileSuffix) && validSuffix.equals(fileSuffix.toLowerCase());
        }

        return false;
    }

    /**
     * 获得指定文件的byte数组
     */
    public static String getFileHeader(String filePath) {
        if (filePath == null || filePath.length() < 1) {
            return null;
        }
        return getFileHeader(new File(filePath));
    }

    public static String getFileHeader(File file) {
        if (file == null) {
            return null;
        }

        FileInputStream fis = null;
        try {
            fis = new FileInputStream(file);
            byte[] b = new byte[4];
            fis.read(b, 0, b.length);

            return bytesToHexString(b);
        } catch (IOException e) {
            if (fis != null) {
                try {
                    fis.close();
                } catch (IOException ignored) {

                }
            }
        }
        return null;
    }

    /**
     * * 将要读取文件头信息的文件的byte数组转换成string类型表示 * * @param src * 要读取文件头信息的文件的byte数组
     * * @return 文件头信息
     */
    private static String bytesToHexString(byte[] src) {
        if (src == null || src.length <= 0) {
            return null;
        }

        StringBuilder builder = new StringBuilder();
        for (byte b : src) {
            // 以十六进制（基数 16）无符号整数形式返回一个整数参数的字符串表示形式，并转换为大写
            String hv = Integer.toHexString(b & 0xFF).toUpperCase();
            if (hv.length() < 2) {
                builder.append(0);
            }
            builder.append(hv);
        }

        return builder.toString();
    }


    public enum FileType {

        JPG("FFD8FF"),
        PNG("89504E47"),
        BMP("424D")
        ;

        private final String header;

        FileType(String header) {
            this.header = header;
        }

        public String getHeader() {
            return header;
        }
    }
}
