package com.hummer.im._internals.log;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;

import com.hummer.im.HMR;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.Locale;
import static android.util.Log.println;
import static com.hummer.im._internals.log.HummerLog.LOG_LEVEL_DEBUG;
import static com.hummer.im._internals.log.HummerLog.LOG_LEVEL_ERROR;
import static com.hummer.im._internals.log.HummerLog.LOG_LEVEL_INFO;
import static com.hummer.im._internals.log.HummerLog.LOG_LEVEL_RELEASE;
import static com.hummer.im._internals.log.HummerLog.LOG_LEVEL_VERBOSE;
import static com.hummer.im._internals.log.HummerLog.LOG_LEVEL_WARN;

/**
 * CIM内部完全依赖Log模块进行日志记录处理，提供了简单、多级的日志记录能力。
 */
public final class Log {

    /**
     * CIM内部完全依赖Log模块进行日志记录处理，提供了简单、多级的日志记录能力。接入业务应在{@link HMR#init}中提供
     * 具体日志记录器的实例。否则只能在本地的Logcat中进行日志输出。
     */
    public interface Logger {

        /**
         * 该注解复用了Android本身的日志分类方式，帮助{@link #log}方法进行level的传值约束
         */
        @IntDef({LOG_LEVEL_VERBOSE, LOG_LEVEL_DEBUG, LOG_LEVEL_INFO, LOG_LEVEL_WARN, LOG_LEVEL_ERROR, LOG_LEVEL_RELEASE})
        @Retention(RetentionPolicy.SOURCE)
        @interface Level {
        }

        /**
         * 进行日志记录
         *
         * @param lv  日志等级，{@link Level}，明确的日志等级对快速日志定位与分类有很大帮助
         * @param msg 日志详情
         */
        void log(@Level int lv, String msg);
    }

    // 在记录日志时，如果每次都用Log.log(Level.Info, "TAGS", "message");
    // 但日志记录属于高频调用，因此下面的工具方法提供了更简单的写法。对上面这条日志来说，可写为：
    // Log.i("TAGS", "message");
    // 不同的日志等级，本别对应不同的工具方法：
    // d: debug, i: info, w: warning, e: err

    public static void v(String tag, Object msg) {
        write(LOG_LEVEL_VERBOSE, tag, msg.toString());
    }

    public static void d(String tag, Object msg) {
        write(LOG_LEVEL_DEBUG, tag, msg.toString());
    }

    public static void i(String tag, Object msg) {
        write(LOG_LEVEL_INFO, tag, msg.toString());
    }

    public static void w(String tag, Object msg) {
        write(LOG_LEVEL_WARN, tag, msg.toString());
    }

    public static void e(String tag, Object msg) {
        write(LOG_LEVEL_ERROR, tag, msg.toString());
    }

    public static void r(String tag, Object msg) {
        write(LOG_LEVEL_RELEASE, tag, msg.toString());
    }

    public static void setLogger(@NonNull final Logger logger) {
        Log.logger = logger;
    }

    public static Logger getLogger() {
        return Log.logger;
    }

    private static void write(@Logger.Level int level, final String tag, final String message) {
        if (logger != null) {
            logger.log(level, stringFrom(tag, message));
        } else {
            println(level, "", stringFrom(tag, message));
        }
    }

    private static String stringFrom(String tag, String message) {
        Thread currentThread = Thread.currentThread();
        String threadInfo = currentThread.getName();
        int pid = android.os.Process.myPid();

        if (tag == null) {
            return String.format(Locale.US, "(%d-%-12s)%s", pid, threadInfo, message, pid);
        } else {
            return String.format(Locale.US, "(%d-%-12s)[%s] %s", pid, threadInfo, tag, message);
        }
    }

    private static Logger logger;
}
