package com.hummer.im._internals.bridge.helper;

import android.text.TextUtils;

import com.hummer.im._internals.bridge.marshall.Marshallable;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.message.TextMessageImpl;
import com.hummer.im.model.chat.AppContent;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.chat.ContentProgress;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.PushContent;
import com.hummer.im.model.chat.contents.Audio;
import com.hummer.im.model.chat.contents.CustomContent;
import com.hummer.im.model.chat.contents.Image;
import com.hummer.im.model.chat.contents.Text;
import com.hummer.im.model.chat.contents.Video;
import com.hummer.im.model.chat.store.MessageStoreStrategy;
import com.hummer.im.model.id.IDFactory;
import com.hummer.im.model.id.User;
import com.hummer.im.model.message.BaseMessage;
import com.hummer.im.model.message.MessageType;
import com.hummer.im.model.user.UserOnlineStatus;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

public class HummerNotification {

    //HUMMER Base Notify

    public static final int NOTIFY_HUMMER_STATE_CHANGED = 100;
    public static final int NOTIFY_HUMMER_KICKED = 101;
    public static final int NOTIFY_HUMMER_TOKEN_EXPIRED = 102;
    public static final int NOTIFY_HUMMER_LOGIN = 103;
    public static final int NOTIFY_HUMMER_REFRESH_TOKEN = 104;
    public static final int NOTIFY_HUMMER_LOGOUT = 105;
    public static final int NOTIFY_HUMMER_REFRESH_TOKEN_1 = 106;
    public static final int NOTIFY_HUMMER_TOKEN_WILL_EXPIRED = 107;
    public static final int NOTIFY_HUMMER_LOG_WITH_LEVEL = 108;

    /// peer callback [160,185]

    public static final int NOTIFY_HUMMER_SEND_P2P_MESSAGE = 160;
    public static final int NOTIFY_HUMMER_FETCH_USER_ONLINE_STATUS = 161;

    /// peer notify [186, 199],

    public static final int NOTIFY_ON_P2P_MESSAGE_RECEIVED = 186;

    public static final int NOTIFY_REPORT_INIT = 200;
    public static final int NOTIFY_REPORT_AREA = 201;
    public static final int NOTIFY_REPORT_METRICS_RETURN_CODE = 202;
    public static final int NOTIFY_REPORT = 203;

    public static final int NOTIFY_ON_SEND_MESSAGE = 301;
    public static final int NOTIFY_ON_BEFORE_SEND_MESSAGE = 302;
    public static final int NOTIFY_ON_AFTER_SEND_MESSAGE = 303;
    public static final int NOTIFY_ON_BEFORE_RECEIVE_MESSAGE = 304;
    public static final int NOTIFY_ON_AFTER_RECEIVE_MESSAGE = 305;
    public static final int NOTIFY_ON_UPDATE_MESSAGE_STATE = 306;
    public static final int NOTIFY_ON_MESSAGE_STATE_UPDATED = 307;
    public static final int NOTIFY_ON_MESSAGE_PREPARE_PROGRESS = 308;
    public static final int NOTIFY_ON_MESSAGE_RECEIVED = 309;

    public static class NotifyBase<T> extends Marshallable {
        private T notification;

        public T get() {
            return notification;
        }

        public void setNotification(T notification) {
            this.notification = notification;
        }
    }

    public static class NotifyBaseCallback extends NotifyBase<BaseCallback> {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            setNotification(new BaseCallback(popInt64(), popInt(), popString16UTF8()));
        }
    }

    public static class BaseCallback {
        private long requestId;
        private int code;
        private String desc;

        public BaseCallback(long requestId, int code, String desc) {
            this.requestId = requestId;
            this.code = code;
            this.desc = desc;
        }

        public long getRequestId() {
            return requestId;
        }

        public void setRequestId(long requestId) {
            this.requestId = requestId;
        }

        public int getCode() {
            return code;
        }

        public void setCode(int code) {
            this.code = code;
        }

        public String getDesc() {
            return desc;
        }

        public void setDesc(String desc) {
            this.desc = desc;
        }
    }

    public static class HummerStateChanged {
        private int oldState;
        private int newState;
        private String reason;

        public HummerStateChanged(int oldState, int newState, String reason) {
            this.oldState = oldState;
            this.newState = newState;
            this.reason = reason;
        }

        public int getOldState() {
            return oldState;
        }

        public int getNewState() {
            return newState;
        }

        public String getReason() {
            return reason;
        }

    }

    public static class HummerLogCallback {
        private int level;
        private String log;

        public HummerLogCallback(int level, String log) {
            this.level = level;
            this.log = log;
        }

        public int getLevel() {
            return level;
        }

        public String getLog() {
            return log;
        }

    }

    public static class HummerKicked {
        private int code;
        private String description;

        public HummerKicked(int code, String description) {
            this.code = code;
            this.description = description;
        }

        public int getCode() {
            return code;
        }

        public String getDescription() {
            return description;
        }
    }

    public static class HummerLogin {
        private long requestId;
        private long userId;
        private int code;
        private String desc;

        public HummerLogin(long requestId, long userId, int code, String desc) {
            this.requestId = requestId;
            this.userId = userId;
            this.code = code;
            this.desc = desc;
        }

        public long getRequestId() {
            return requestId;
        }

        public long getUserId() {
            return userId;
        }

        public int getCode() {
            return code;
        }

        public String getDesc() {
            return desc;
        }
    }

    public static class HummerRefreshToken {
        private long requestId;
        private int code;
        private String desc;

        public HummerRefreshToken(long requestId, int code, String desc) {
            this.requestId = requestId;
            this.code = code;
            this.desc = desc;
        }

        public long getRequestId() {
            return requestId;
        }

        public int getCode() {
            return code;
        }

        public String getDesc() {
            return desc;
        }
    }

    public static class HummerLogout {
        private long requestId;
        private int code;
        private String desc;

        public HummerLogout(long requestId, int code, String desc) {
            this.requestId = requestId;
            this.code = code;
            this.desc = desc;
        }

        public long getRequestId() {
            return requestId;
        }

        public int getCode() {
            return code;
        }

        public String getDesc() {
            return desc;
        }
    }

    public static class ReportInit {
        private String appKey;
        private String version;
        private String from;

        public ReportInit(String appKey, String version, String from) {
            this.appKey = appKey;
            this.version = version;
            this.from = from;
        }

        public String getAppKey() {
            return appKey;
        }

        public String getVersion() {
            return version;
        }

        public String getFrom() {
            return from;
        }
    };


    public static class ReportArea {
        private int area;

        public ReportArea(int area) {
            this.area = area;
        }

        public int getArea() {
            return area;
        }
    }

    public static class ReportMetricsReturnCode {
        private int scode;
        private String uri;
        private long timeConsumption;
        private String code;

        public ReportMetricsReturnCode(int scode, String uri, long timeConsumption, String code) {
            this.scode = scode;
            this.uri = uri;
            this.timeConsumption = timeConsumption;
            this.code = code;
        }

        public int getSCode() {
            return scode;
        }

        public String getUri() {
            return uri;
        }

        public long getTimeConsumption() {
            return timeConsumption;
        }

        public String getCode() {
            return code;
        }
    }

    public static class Report {
        private String type;
        private Map<String, String> data;

        public Report(String type, Map<String, String> data) {
            this.type = type;
            this.data = data;
        }

        public String getType() {
            return type;
        }

        public Map<String, String> getData() {
            return data;
        }
    }

    public static class OnMessage {
        private Message message;
        private int msgType;
        private long userId;
        private long roomId;
        private String data;

        public OnMessage(Message message, int msgType, long userId, long roomId, String data) {
            this.message = message;
            this.msgType = msgType;
            this.userId = userId;
            this.roomId = roomId;
            this.data = data;
        }

        public Message getMessage() {
            return message;
        }

        public int getMsgType() {
            return msgType;
        }

        public void setMsgType(int msgType) {
            this.msgType = msgType;
        }

        public long getUserId() {
            return userId;
        }

        public void setUserId(long userId) {
            this.userId = userId;
        }

        public long getRoomId() {
            return roomId;
        }

        public void setRoomId(long roomId) {
            this.roomId = roomId;
        }

        public String getData() {
            return data;
        }

        public void setData(String data) {
            this.data = data;
        }
    }

    public static class OnSendMessage {
        private long requestId;
        private int code;
        private String desc;
        private Map<String, String> extraInfo;
        private Message message;

        public OnSendMessage(long requestId, int code, String desc, Map<String, String> info, Message message) {
            this.requestId = requestId;
            this.code = code;
            this.desc = desc;
            this.extraInfo = info;
            this.message = message;
        }

        public long getRequestId() {
            return requestId;
        }

        public int getCode() {
            return code;
        }

        public String getDesc() {
            return desc;
        }

        public Message getMessage() {
            return message;
        }

        public Map<String, String> getExtraInfo() { return extraInfo; }

    }

    public static class NotifyOnFetchUserOnlineStatus extends NotifyBase<OnFetchUserOnlineStatus>{

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long requestId = popInt64();
            int code = popInt();
            String desc = popString16UTF8();
            Map<Long, Integer> statusMap = popMap(Long.class, Integer.class);
            Set<UserOnlineStatus> status = new HashSet<>(statusMap.size());
            for (Map.Entry<Long, Integer> entry : statusMap.entrySet()) {
                status.add(new UserOnlineStatus(new User(entry.getKey()),
                        UserOnlineStatus.OnlineStatus.getOnlineStatus(entry.getValue())));
            }

            setNotification(new OnFetchUserOnlineStatus(requestId, code, desc, status));
        }
    }

    public static class OnFetchUserOnlineStatus extends BaseCallback{

        private Set<UserOnlineStatus> status;

        public OnFetchUserOnlineStatus(long requestId, int code, String desc, Set<UserOnlineStatus> status) {
            super(requestId, code, desc);
            this.status = status;
        }

        public Set<UserOnlineStatus> getStatus() {
            return status;
        }
    }

    public static class NotifyOnP2PMessageReceived extends NotifyBase<OnP2PMessageReceived> {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long userId = popInt64();
            String uuid = popString16UTF8();
            long ts = popInt64();
            int msgType = popInt();
            Map<String, String> extra = popMap(String.class, String.class);
            BaseMessage message;
            if (msgType == MessageType.TEXT.getType()) {
                String text = popString32("utf-8");
                message = new TextMessageImpl(uuid, ts, text);
                message.setExtra(extra);
            } else {
                message = null;
            }

            setNotification(new OnP2PMessageReceived(new User(userId), message));
        }
    }

    public static class OnP2PMessageReceived {
        private User user;
        private BaseMessage baseMessage;

        public OnP2PMessageReceived(User user, BaseMessage baseMessage) {
            this.user = user;
            this.baseMessage = baseMessage;
        }

        public User getUser() {
            return user;
        }

        public BaseMessage getBaseMessage() {
            return baseMessage;
        }
    }

    public static class NotifyOnMessageStateUpdated extends NotifyOnMessage<OnMessageStateUpdated> {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long requestId = popInt64();
            setNotification(new OnMessageStateUpdated(requestId, getMessage()));
        }
    }

    public static class OnMessageStateUpdated {
        private final long requestId;
        private final Message message;

        public OnMessageStateUpdated(long requestId, Message message) {
            this.requestId = requestId;
            this.message = message;
        }

        public Message getMessage() {
            return message;
        }

        public long getRequestId() {
            return requestId;
        }
    }

    public static class NotifyOnMessagePrepareProgress extends NotifyOnMessage<OnMessagePrepareProgress> {
        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long requestId = popInt64();
            long total = popInt64();
            long current = popInt64();
            setNotification(new OnMessagePrepareProgress(requestId, getMessage(), total, current));
        }
    }

    public static class OnMessagePrepareProgress {
        private final long requestId;
        private final Message message;

        public OnMessagePrepareProgress(long requestId, Message message, long total, long current) {
            this.requestId = requestId;
            ContentProgress progress = new ContentProgress(total, current);
            message.getContent().setProgress(progress);
            this.message = message;
        }

        public Message getMessage() {
            return message;
        }

        public long getRequestId() {
            return requestId;
        }
    }

    public static class NotifyOnMessageReceived extends NotifyOnMessage<OnMessageReceived> {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            setNotification(new OnMessageReceived(getMessage()));
        }
    }

    public static class OnMessageReceived {
        private final Message message;

        public OnMessageReceived(Message message) {
            this.message = message;
        }

        public Message getMessage() {
            return message;
        }
    }

    public static class NotifyOnMessage<T> extends NotifyBase<T> {

        private Message message;

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);

            int stateType = popInt();
            int stateCode = popInt();
            String stateDesc = popString16UTF8();
            Map<String, String> extraInfo = popMap(String.class, String.class);
            String uuid = popString16UTF8();
            long timestamp = popInt64();
            String sender = popString16UTF8();
            String receiver = popString16UTF8();
            String extra = popString16UTF8();
            Map<String, String> kvExtra = popMap(String.class, String.class);
            int msgType = popInt();
            Content content;
            if (msgType == Text.TYPE) {
                content = new Text(popString32("UTF-8"));
            } else if (msgType == Image.TYPE) {
                String originUrl = popString16UTF8();
                String thumbUrl = popString16UTF8();
                int width = popInt();
                int height = popInt();
                Image image = Image.create(originUrl, width, height);
                image.setThumbUrl(thumbUrl);

                content = image;
            } else if (msgType == Audio.TYPE) {
                String url = popString16UTF8();
                int duration  = popInt();
                content = Audio.create(url, duration);

            } else if (msgType == Video.TYPE) {
                String videoUrl = popString16UTF8();
                int videoSize = popInt();
                int videoWidth = popInt();
                int videoHeight = popInt();
                int videoDuration = popInt();
                String coverUrl = popString16UTF8();
                String coverThumbUrl = popString16UTF8();
                int coverWidth = popInt();
                int coverHeight = popInt();

                Video video = Video.create(videoUrl, videoSize, videoWidth, videoHeight, videoDuration, coverUrl, coverWidth, coverHeight);
                video.setCoverThumbUrl(coverThumbUrl);
                content = video;

            } else if (msgType == CustomContent.TYPE) {
                content = new CustomContent(popBytes32());
            } else if (msgType >= CustomContent.CustomTypeStart) {
                content = new AppContent(msgType, popBytes32());
            } else {
                Log.e("HummerNotification", "msgType cannot parse, msgType: " + msgType);
                content = null;
            }

            boolean isRemoteHistoryMessage = popBool();
            boolean isLocalHistoryMessage = popBool();
            boolean isLastMessage = popBool();
            boolean isUnreadCount = popBool();
            MessageStoreStrategy storeStrategy
                    = new MessageStoreStrategy(isRemoteHistoryMessage, isLocalHistoryMessage, isLastMessage, isUnreadCount);

            String pushTitle = popString32("UTF-8");
            String pushContent = popString32("UTF-8");
            String pushIconUrl = popString32("UTF-8");
            byte[] pushPayload = popBytes32();
            PushContent push = null;
            if (!TextUtils.isEmpty(pushTitle)
                    || !TextUtils.isEmpty(pushContent)
                    || !TextUtils.isEmpty(pushIconUrl)
                    || (pushPayload != null && pushPayload.length > 0)) {
                push = new PushContent(pushTitle, pushContent, pushPayload, pushIconUrl);
            }

            Message message = new Message();

            message.setState(Message.buildSate(stateType, stateCode, stateDesc, extraInfo));
            message.setUuid(uuid);
            message.setTimestamp(timestamp);
            message.setSender(IDFactory.makeId(sender));
            message.setReceiver(IDFactory.makeId(receiver));
            message.setAppExtra(extra);
            message.setKvExtra(kvExtra);
            message.setContent(content);
            message.setStoreStrategy(storeStrategy);
            message.setPushContent(push);

            this.message = message;
        }

        public Message getMessage() {
            return message;
        }
    }
}
