package com.hummer.im._internals.bridge.helper;

import android.util.Log;

import com.hummer.im._internals.ChatServiceImpl;
import com.hummer.im._internals.bridge.marshall.Marshallable;
import com.hummer.im._internals.bridge.marshall.MshBuffer;
import com.hummer.im._internals.bridge.marshall.NativeByteBufferPool;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.MessageOptions;
import com.hummer.im.model.message.TextMessage;
import com.hummer.im.model.option.FetchStrategy;
import com.hummer.im.model.option.HummerOptions;

import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

import static com.hummer.im._internals.bridge.helper.HummerEvent.EVENT_REFRESH_BYTE_TOKEN_1;
import static com.hummer.im._internals.bridge.helper.HummerEvent.EVENT_REFRESH_TOKEN_1;
import static com.hummer.im._internals.bridge.helper.HummerEvent.EVENT_SET_DEBUG_MODE;
import static com.hummer.im._internals.bridge.helper.HummerEvent.EVENT_SET_LOG_CALLBACK;

public class HummerNative {

    private final static Set<NotificationListener> mNotificationListeners = new CopyOnWriteArraySet<>();

    public interface NotificationListener {

        /**
         * 事件的通知回调
         *
         * @param type 通知的类型标识
         * @param data 收到的回调数据
         */
        void handleNotify(int type, byte[] data);

    }

    public static void registerNotificationListener(NotificationListener listener) {
        if (listener != null) {
            synchronized (mNotificationListeners) {
                mNotificationListeners.add(listener);
            }
        }
    }

    /**
     * C++ 代码通知回调函数
     */
    public static void notificationCallback(int type, byte[] data) {

        /* 单独处理日志回调，避免在统一处理的地方有打印日志，导致死循环 */
        if (type == HummerNotification.NOTIFY_HUMMER_LOG_WITH_LEVEL) {
            HummerEngine.getInstance().handleLogCallback(data);
            return;
        }

        com.hummer.im._internals.log.Log.i(TAG, "handleNotify | type: " + type);

        synchronized (mNotificationListeners) {
            for (NotificationListener listener : mNotificationListeners) {
                listener.handleNotify(type, data);
            }
        }
    }

    /******************************** native begin *****************************************/


    private static native long nativeInit(byte[] array);

    private native static String nativeSdkGetVersion();

    private native static String nativeMakeUUID();

    private native static long nativeGetSyncTs();

    private native static int nativeSdkGetState(long requestId);

    public native static TextMessage nativeCreateTextMessage(String text);

//    private native static long nativeLogout();

    public native static long nativeAddMessageListener();

    public native static long nativeRemoveMessageListener();

    public native static long nativeAddMemberListener();

    public native static long nativeRemoveMemberListener();

    public native static long nativeAddRoomListener();

    public native static long nativeRemoveRoomListener();

    public native static long nativeSdkProcess(int event, byte[] data);


    /******************************** native end *****************************************/

    public static long sdkProcess(HummerEvent.EventBase e) {
        MshBuffer buffer = new MshBuffer(Marshallable.kProtoPacketSize, NativeByteBufferPool.get());
        long ret = nativeSdkProcess(e.event(), e.HPmarshall(buffer));
        buffer.freeBuffer();
        return ret;
    }

    public static long init(long addId, String appVersion, boolean isDelegateMode) {
        HummerEvent.EventInitSdk event = new HummerEvent.EventInitSdk(addId, appVersion, isDelegateMode);
        MshBuffer buffer = new MshBuffer(Marshallable.kProtoPacketSize, NativeByteBufferPool.get());
        byte[] data = event.HPmarshall(buffer);
        long ret = nativeInit(data);
        buffer.freeBuffer();
        return ret;
    }

    public static long login(long requestId, long uid, String region, byte[] token, HummerOptions options) {
        HummerEvent.EventLoginByte event = new HummerEvent.EventLoginByte(requestId, uid, region, token, options);
        return sdkProcess(event);
    }

    public static long logout(long requestId) {
        HummerEvent.EventLogout event = new HummerEvent.EventLogout(requestId);
        long ret = sdkProcess(event);
        return ret;
    }

    public static String getSdkVersion() {
        return nativeSdkGetVersion();
    }

    public static String makeUUID() {
        return nativeMakeUUID();
    }

    public static long getSyncTs() {
        return nativeGetSyncTs();
    }

    public static int getState(long requestId) {
        return nativeSdkGetState(requestId);
    }

    public static long refreshToken(long requestId, String token, HummerOptions options) {
        HummerEvent.EventRefreshToken event = new HummerEvent.EventRefreshToken(requestId, token, options);
        return sdkProcess(event);
    }

    public static long refreshToken1(long requestId, String token, HummerOptions options) {
        HummerEvent.EventRefreshToken event = new HummerEvent.EventRefreshToken(requestId, token, options);
        event.event = EVENT_REFRESH_TOKEN_1;
        return sdkProcess(event);
    }

    public static long refreshToken1(long requestId, byte[] token, HummerOptions options) {
        HummerEvent.EventRefreshByteToken event = new HummerEvent.EventRefreshByteToken(requestId, token, options);
        event.event = EVENT_REFRESH_BYTE_TOKEN_1;
        return sdkProcess(event);
    }

    public static long logText(int level, int module, String moduleName, String tag, String text) {
        return sdkProcess(new HummerEvent.EventLogText(level, module, moduleName, tag, text));
    }

    public static long setLogFilePath(String filePath) {
        return sdkProcess(new HummerEvent.EventLogFilePath(filePath));
    }

    public static long setLogLevel(int level) {
        return sdkProcess(new HummerEvent.EventLogSetLevel(level));
    }

    public static long disableLogConsole(boolean disable) {
        return sdkProcess(new HummerEvent.EventLogDisableConsole(disable));
    }

    public static long uploadLog(String note) {
        return sdkProcess(new HummerEvent.EventUploadLog(note));
    }

    public static long setDebugMode(boolean enable) {
        HummerEvent.EventLogDisableConsole event = new HummerEvent.EventLogDisableConsole(enable);
        event.event = EVENT_SET_DEBUG_MODE;
        return sdkProcess(event);
    }

    public static long setLogCallback(boolean enable) {
        HummerEvent.EventLogDisableConsole event = new HummerEvent.EventLogDisableConsole(enable);
        event.event = EVENT_SET_LOG_CALLBACK;
        return sdkProcess(event);
    }

    public static long setFetchStrategy(FetchStrategy strategy) {
        return sdkProcess(new HummerEvent.EventSetFetchStrategy(strategy));
    }

    public static long sendMessage(long requestId, ChatServiceImpl.MessagePacket packet) {
        return sdkProcess(new HummerEvent.EventSendMessage(requestId, packet));
    }

    public static void sendMessage(long requestId, Message message, MessageOptions options) {
        sdkProcess(new HummerEvent.EventSendMessageWithOptions(requestId, message, options));
    }

    public static long reportReturnCode(String funcName, long rtt, int code) {
        return sdkProcess(new HummerEvent.EventReportReturnCode(funcName, rtt, code));
    }

    private static final String TAG = "HummerNativeJ";

    static {
        try {
            System.loadLibrary("core");
        } catch (Throwable e) {
            e.printStackTrace();
            Log.e(TAG, "load core sdk failed!");
        }

    }

}
