package com.hummer.im._internals.bridge.helper;

import com.hummer.im._internals.ChatServiceImpl;
import com.hummer.im._internals.bridge.marshall.HPMarshaller;
import com.hummer.im._internals.chatsvc.IdentifiableHelper;
import com.hummer.im.model.chat.AppContent;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.MessageOptions;
import com.hummer.im.model.chat.contents.Audio;
import com.hummer.im.model.chat.contents.CustomContent;
import com.hummer.im.model.chat.contents.Image;
import com.hummer.im.model.chat.contents.Text;
import com.hummer.im.model.chat.contents.Video;
import com.hummer.im.model.id.IDFactory;
import com.hummer.im.model.id.User;
import com.hummer.im.model.message.BaseMessage;
import com.hummer.im.model.message.P2PMessageOptions;
import com.hummer.im.model.message.TextMessage;
import com.hummer.im.model.option.FetchStrategy;
import com.hummer.im.model.option.HummerOptions;
import com.hummer.im.model.option.TokenType;

import java.util.HashSet;
import java.util.Set;

public class HummerEvent {

    /********************************  Events Begin *****************************************/
    //HUMMER Base Events
    private static final int EVENT_SDK_INIT = 101;
    private static final int EVENT_LOGOUT = 103;
    private static final int EVENT_REFRESH_TOKEN = 104;
    private static final int EVENT_LOG_TEXT = 105;
    private static final int EVENT_LOG_FILE_PATH = 106;
    private static final int EVENT_SET_LOG_LEVEL = 107;
    private static final int EVENT_LOG_DISABLE_CONSOLE = 108;
    private static final int EVENT_UPLOAD_LOG = 109;
    public static final int EVENT_SET_DEBUG_MODE = 110;
    public static final int EVENT_SET_LOG_CALLBACK = 111;
    public static final int EVENT_SET_FETCH_STRATEGY = 112;
    public static final int EVENT_LOGIN_BYTE = 113;

    private static final int EVENT_SEND_MESSAGE = 201;
    public static final int EVENT_REFRESH_TOKEN_1 = 202;
    private static final int EVENT_REPORT_METRICS_RETURN_CODE = 203;
    private static final int EVENT_SEND_MESSAGE_WITH_OPTIONS = 204;
    public static final int EVENT_REFRESH_BYTE_TOKEN_1 = 205;

    private static final int EVENT_SEND_P2P_MESSAGE = 250;
    private static final int EVENT_FETCH_USER_ONLINE_STATUS = 251;

    public static class EventBase extends HPMarshaller {
        public int event = 0;

        public int event() {
            return event;
        }

        @Override
        public byte[] marshall() {
            return super.marshall();
        }
    }

    public static class EventInitSdk extends EventBase {
        private final long appId;
        private final String appVersion;
        private final boolean isDelegateMode;

        EventInitSdk(long appId, String appVersion, boolean isDelegateMode) {
            event = EVENT_SDK_INIT;
            this.appId = appId;
            this.appVersion = appVersion;
            this.isDelegateMode = isDelegateMode;
        }

        @Override
        public byte[] marshall() {
            pushInt64(appId);
            pushString16(appVersion);
            pushBool(isDelegateMode);

            return super.marshall();
        }
    }

    public static class EventLoginByte extends EventBase {
        private final long requestId;
        private final long uid;
        private final String region;
        private final byte[] token;
        private final HummerOptions options;

        EventLoginByte(long requestId, long uid, String region, byte[] token, HummerOptions options) {
            event = EVENT_LOGIN_BYTE;
            this.requestId = requestId;
            this.uid = uid;
            this.region = region;
            this.token = token;
            this.options = options;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushInt64(uid);
            pushString16(region);
            pushBytes32(token);
            if (options == null) {
                pushInt(TokenType.THIRD_USER_TOKEN.getType());
            } else {
                pushInt(options.getTokenType().getType());
            }

            return super.marshall();
        }
    }

    public static class EventLogout extends EventBase {
        private long requestId;

        EventLogout(long requestId) {
            event = EVENT_LOGOUT;
            this.requestId = requestId;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            return super.marshall();
        }
    }

    public static class EventLogText extends EventBase {
        private int level;
        private int module;
        private String moduleName;
        private String tag;
        private String text;

        EventLogText(int _level, int _module, String _moduleName, String _tag, String _text) {
            event = EVENT_LOG_TEXT;
            level = _level;
            module = _module;
            moduleName = _moduleName;
            tag = _tag;
            text = _text;
        }

        @Override
        public byte[] marshall() {
            pushInt(level);
            pushInt(module);
            pushString16(moduleName);
            pushString16(tag);
            pushString32(text);
            return super.marshall();
        }
    }

    public static class EventLogFilePath extends EventBase {
        private String filePath;

        EventLogFilePath(String _filePath) {
            event = EVENT_LOG_FILE_PATH;
            filePath = _filePath;
        }

        @Override
        public byte[] marshall() {
            pushString16(filePath);
            return super.marshall();
        }
    }

    public static class EventLogSetLevel extends EventBase {
        private int level;

        EventLogSetLevel(int _level) {
            event = EVENT_SET_LOG_LEVEL;
            level = _level;
        }

        @Override
        public byte[] marshall() {
            pushInt(level);
            return super.marshall();
        }
    }

    public static class EventLogDisableConsole extends EventBase {
        private boolean disable;

        EventLogDisableConsole(boolean _disable) {
            event = EVENT_LOG_DISABLE_CONSOLE;
            disable = _disable;
        }

        @Override
        public byte[] marshall() {
            pushBool(disable);
            return super.marshall();
        }
    }

    public static class EventUploadLog extends EventBase {
        private String note;

        EventUploadLog(String _note) {
            event = EVENT_UPLOAD_LOG;
            note = _note;
        }

        @Override
        public byte[] marshall() {
            pushString16(note);
            return super.marshall();
        }
    }

    public static class EventSetFetchStrategy extends EventBase {
        private final FetchStrategy strategy;

        public EventSetFetchStrategy(FetchStrategy strategy) {
            event = EVENT_SET_FETCH_STRATEGY;
            this.strategy = strategy;
        }

        @Override
        public byte[] marshall() {
            switch (strategy) {
                case CONTINUOUSLY:
                    pushInt(1);
                    break;
                case SKIP_FETCHED:
                    pushInt(2);
                    break;
                case RELOAD_HISTORIES:
                    pushInt(3);
                    break;
                case IGNORE_BEFORE:
                    pushInt(4);
                    break;
            }

            return super.marshall();
        }
    }

    public static class EventRefreshToken extends EventBase {
        private final long requestId;
        private final String token;
        private final HummerOptions options;

        EventRefreshToken(long requestId, String token, HummerOptions options) {
            event = EVENT_REFRESH_TOKEN;
            this.requestId = requestId;
            this.token = token;
            this.options = options;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushString16(token);
            if (options == null) {
                pushInt(TokenType.THIRD_USER_TOKEN.getType());
            } else {
                pushInt(options.getTokenType().getType());
            }
            return super.marshall();
        }
    }

    public static class EventRefreshByteToken extends EventBase {
        private final long requestId;
        private final byte[] token;
        private final HummerOptions options;

        EventRefreshByteToken(long requestId, byte[] token, HummerOptions options) {
            event = EVENT_REFRESH_TOKEN;
            this.requestId = requestId;
            this.token = token;
            this.options = options;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushBytes32(token);
            if (options == null) {
                pushInt(TokenType.THIRD_USER_TOKEN.getType());
            } else {
                pushInt(options.getTokenType().getType());
            }
            return super.marshall();
        }
    }

    public static class EventSendMessage extends EventBase {
        private long requestId;
        private ChatServiceImpl.MessagePacket packet;

        EventSendMessage(long requestId, ChatServiceImpl.MessagePacket packet) {
            event = EVENT_SEND_MESSAGE;
            this.requestId = requestId;
            if (packet != null) {
                this.packet = packet;
            } else {
                this.packet = new ChatServiceImpl.MessagePacket();
            }
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushString16(packet.getUuid());
            pushInt(packet.getMsgType());
            pushInt64(packet.getUserId());
            pushInt(packet.getRoomId());
            pushString32(packet.getData(), "utf-8");
            pushString16(packet.getExtra());
            pushMap(packet.getKvExtra(), String.class);
            return super.marshall();
        }
    }

    public static class EventSendMessageWithOptions extends EventBase {
        private long requestId;
        private Message message;
        private MessageOptions options;

        EventSendMessageWithOptions(long requestId, Message message, MessageOptions options) {
            event = EVENT_SEND_MESSAGE_WITH_OPTIONS;
            this.requestId = requestId;
            this.message = message;
            this.options = options;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushString16(this.message.getUuid());
            if (message.getContent() instanceof Text) {
                pushInt(Text.TYPE);
                pushString32(((Text) message.getContent()).getText(), "UTF-8");

            } else if (message.getContent() instanceof Image) {
                pushInt(Image.TYPE);
                Image image = (Image) message.getContent();
                pushString16(image.getOriginUrl());
                pushInt(image.getWidth());
                pushInt(image.getHeight());

            } else if (message.getContent() instanceof Audio) {
                pushInt(Audio.TYPE);
                Audio audio = (Audio) message.getContent();
                pushString16(audio.getUrl());
                pushInt(audio.getDuration());

            } else if (message.getContent() instanceof Video) {
                pushInt(Video.TYPE);
                Video video = (Video) message.getContent();
                pushString16(video.getVideoUrl());
                pushInt(video.getVideoSize());
                pushInt(video.getVideoWidth());
                pushInt(video.getVideoHeight());
                pushInt(video.getVideoDuration());
                pushString16(video.getCoverUrl());
                pushInt(video.getCoverWidth());
                pushInt(video.getCoverHeight());

            } else if (message.getContent() instanceof CustomContent) {
                pushInt(CustomContent.TYPE);
                pushBytes32(((CustomContent) message.getContent()).getData());

            } else if (message.getContent() instanceof AppContent) {
                pushInt(((AppContent) message.getContent()).getType());
                pushBytes32(((AppContent) message.getContent()).getData());

            }

            pushString16(IDFactory.makeType(message.getReceiver()).getType());
            pushString32(IDFactory.makeString(message.getReceiver()));

            pushString16(message.getAppExtra());
            pushMap(message.getKvExtra(), String.class);

            pushBool(message.getStoreStrategy().isRemoteHistoryMessage());
            pushBool(message.getStoreStrategy().isLocalHistoryMessage());
            pushBool(message.getStoreStrategy().isLastMessage());
            pushBool(message.getStoreStrategy().isUnreadCount());

            if (message.getPushContent() != null) {
                pushBool(true);
                pushString32(message.getPushContent().getContent());
                pushString32(message.getPushContent().getTitle());
                pushString32(message.getPushContent().getIconUrl());
                pushBytes32(message.getPushContent().getPayload());
            } else {
                pushBool(false);
            }

            return super.marshall();
        }
    }

    public static class EventReportReturnCode extends EventBase {
        private String funcName;
        private long rtt;
        private int code;

        EventReportReturnCode(String funcName, long rtt, int code) {
            event = EVENT_REPORT_METRICS_RETURN_CODE;
            this.funcName = funcName;
            this.rtt = rtt;
            this.code = code;
        }

        @Override
        public byte[] marshall() {
            pushString16(funcName);
            pushInt64(rtt);
            pushInt(code);
            return super.marshall();
        }
    }

    public static class EventSentP2PMessage extends EventBase {
        private long requestId;
        private User receiver;
        private P2PMessageOptions options;
        private BaseMessage message;

        public EventSentP2PMessage(long requestId, User receiver, P2PMessageOptions options, BaseMessage message) {
            event = EVENT_SEND_P2P_MESSAGE;
            this.requestId = requestId;
            this.receiver = receiver;
            this.options = options;
            this.message = message;
        }

        @Override
        public byte[] marshall() {
            pushInt64(requestId);
            pushInt64(receiver == null ? 0 : receiver.getId());
            pushBool(options != null && options.getOffline());
            pushString16(message.getUuid());
            pushInt64(message.getTimestamp());
            pushInt(message.getMessageType().getType());
            pushMap(message.getExtra(), String.class);
            if (message instanceof TextMessage) {
                pushString32(((TextMessage) message).getText(), "UTF-8");
            }

            return super.marshall();
        }
    }

    public static class EventFetchUserOnlineStatus extends EventBase {
        private long requestId;
        private Set<User> users;

        public EventFetchUserOnlineStatus(long requestId, Set<User> users) {
            event = EVENT_FETCH_USER_ONLINE_STATUS;
            this.requestId = requestId;
            this.users = users;
        }

        @Override
        public byte[] marshall() {
            if (users == null) {
                users = new HashSet<>();
            }
            pushInt64(requestId);
            pushInt(users.size());
            for (User user : users) {
                pushInt64(user.getId());
            }

            return super.marshall();
        }
    }

}

